/*
 *  Description: Host serial slave
 *
 *  Designer : Paul Likoudis
 *  Company  : VLSI Vision Ltd.
 *  Module   : ss653a
 *
 * Revision History:
 * DD-MMM-YY:  Comment
 * 21-MAY-97:  Copy from 404 original (by PL)
 * 22-MAY-97:  Single clock design
*/

module ss653a(
  clk14,
  rstxxa_n,
  sda,
  scl,
  din,
  dout,
  ale,
  wrb,
  vpen,
  sad,
  ss_din,
  ss_dout,
  ss_vpen,
  ss_ale,
  ss_wrb,
  hsda_,
  active,
  sad_
);
  
  input clk14;           // Input clock
  input rstxxa_n;        // Input reset (bar)
  input sda;             // Serial comms data
  input scl;             // Serial comms clock
  input [7:0] din;       // Bus with address/data for parallel interface
  input [7:0] ss_dout;   // Data from Reg. file
  input ale;             // Address Latch Enable (from micro)
  input wrb;             // Write (bar) pulse (from micro)
  input vpen;            // config pin, zero enables SIF
  input sad;		 // fst pad input side

  output [7:0] ss_din;   // data towards REG. File
  output [7:0] dout;     // data towards pins
  output ss_vpen;        // used to access register file.
  output ss_ale;         // Address Latch Enable (to reg.file)
  output ss_wrb;         // Write (bar) pulse (to reg.file)
  output hsda_;          // Serial comms data bit (to gen acks)
  output active;         // Flag to indicate sif message in progress
  output sad_;		 // OEN for fst/sad pin

  reg [7:0] ss_din, dout;
  reg ss_vpen;
  reg ss_ale;
  reg ss_wrb;

  reg [6:0] ix;          // Address from serial interface register map
  reg inc;		 // Auto increment flag
  reg [7:0] shifty;	 // ser <-> par
  reg [5:0] bit_count;   // state register (+1)
  reg [1:0] sda_, scl_;  // input filters, edge detectors
  reg hsda_,		 // ack (pulls sda low)
      active,		 // high between start and stop
      rwbit,		 // ...
      sad1,		 // slave address bit 1, from fst @ por
      sad_,		 // output enable for fst pad
      addd;		 // high if address matches

  wire ack;

  always @ (posedge clk14 or negedge rstxxa_n)
    if (~rstxxa_n) begin
      {shifty, bit_count,hsda_,addd,ss_wrb,dout,sad_} <= ~0;
      {inc,ix, sda_,scl_,active,rwbit,ss_vpen,ss_din,ss_ale,sad1} <= 0;
    end
    else begin 
      sda_ <= {sda_, sda};		 // input edge detectors and filters
      scl_ <= {scl_, scl};
      {sad_,sad1} <= sad_? sad : sad1;  // sad <= 0 ;-)
      if (scl_ == 'b11 && ^sda_ || bit_count == 63) begin 
	bit_count <= 0;
	hsda_ <= 1;
	rwbit<=0;
	addd<=1;
	active <= scl_ == 'b11 && sda_ == 'b10;	 // START
      end
      else if (addd && scl_ == 'b01) begin 
	case (bit_count)
	  7: begin
	    if (shifty[6:0] != ((`SS_ADDR>>1) ^ sad1))
	      addd <= 0;
	    bit_count <= 8;
	    shifty <= {shifty, sda_[1]};
	  end
	  8: begin
	    rwbit <= shifty[0];		 // to allow byte#0 ack
	    bit_count <= 9;
	    shifty <= {inc,ix};
	  end
	  16: begin
            if (~rwbit)
	      {inc,ix} <= {shifty, sda_[1]};	// host sends messages 
	    bit_count <= 17;
            shifty <= {shifty, sda_[1]};
          end
	  17: begin 
            ix <= ix + (inc & rwbit);
	    bit_count <= (rwbit & sda_[1]) ? ~0 : 18;
            if (rwbit)
	      shifty <= ss_dout;
	  end
	  26: begin 
            ix <= ix + inc;
	    bit_count <= ~sda_[1] ? 18 : ~0;
            if (rwbit)
	      shifty <= ss_dout;
	  end
	  default: begin 
	    bit_count <= addd ? bit_count + ~&bit_count  : 0;
	    shifty <= {shifty, sda_[1]};
	  end
        endcase
      end
      else if (addd && scl_ == 'b10)begin 
        hsda_ <=  (shifty[7]| ack | ~rwbit) & ~(ack & ~rwbit);
      end
      if (vpen) begin 
        ss_vpen <= 1; ss_ale <= ale; ss_din <= din; dout <= ss_dout;
	ss_wrb <= wrb;
      end
      else begin 
        ss_vpen <= active;
        case (bit_count)
	  17: begin 
	    ss_ale <= scl_ == 0;
	    ss_din <= ix;
	  end
	  26: begin 
	    ss_ale <= rwbit && scl_ == 0;
	    ss_wrb <= !(scl_ == 0 && !rwbit);
	    ss_din <= rwbit ? ix : shifty;
	  end
	  default: {ss_ale,ss_wrb} <= 2'b01;
	endcase
      end
    end

/*
 * Data out and ACK generator: always @9, if write, @18,27 and never when
 *  writing.
 */
  assign ack = (bit_count == 8 || bit_count == 17 || bit_count == 26);

endmodule


