/*
 *  Description: Defect RAM loader/reader
 *
 *  Designer: Andrew Murray & Mairi Torrie
 *  Company  : VLSI Vision Ltd.
 *  Module   : dc653a
 *
 * Revision History:
 * DD-MMM-YY:  Comment
 * 21-MAY-97:  Copy from CPiA original
*/


module dc653a (
  vp_clk,
  resetb,
  map,
  sof,
  data_valid_pre,
  data_valid_posen,
  ram_data,
  ram_waddress,
  ram_wceb,
  e2read,
  ram_address,
  ram_ceb,
  ram_web,
  correct,
  filter_code
);

   input vp_clk;
   input resetb;
   input map;
   input sof;
   input data_valid_pre;
   input data_valid_posen;
   input [15:0] ram_data;
   input [6:0] ram_waddress;
   input ram_wceb;
   input e2read;
   output [6:0] ram_address;
   output ram_ceb;
   output ram_web;
   output correct;
   output [1:0] filter_code;


   wire [6:0] ram_address;
   wire ram_ceb;
   reg correct;
   reg [1:0] filter_code;

   reg [6:0] ram_raddress;
   wire ram_rceb;
   reg [1:0] filter_codei;
   reg [3:0] state;
   reg [3:0] nextstate;
   reg jump, jumpd, jumpd2;
   reg [13:0] distance_left;
   wire apply_filter;
   reg addr_event;
   reg addr_event_pre;
   reg addr_event_posen;
   reg distance_event;
   reg distance_event_pre;
   reg distance_event_posen;
   wire distance_load_enable;
   wire distance_zero;


   assign ram_address = e2read ? ram_waddress : ram_raddress;
   assign ram_ceb = e2read ? ram_wceb : ram_rceb;
   assign ram_web = e2read ? 0 : 1;

/* The controlling state machine 
 *
 * This version of the machine uses gray code. Field level sync is made using 
 * the presence of sof, sync of the two-state cycle and the pixel stream is 
 * made using the precise timing of sof.
 * If the the relationship between sof and the array timing changes be an odd
 * number of vp_clk periods, this can be allowed for in two ways: either
 * sacrificing a gray transition in the state machine, moving directly from
 * 2 to 7 (rather than 6) (and dealing with any consequences), or altering the
 * re-timing used on 'correct' and 'filter_code'. In addition, any change in
 * sof/array timing may require an alteration in the latency adjustment made
 * to the first defect distance (when making up the defect map).
 * The exits from both states 4 and 7 are dependent on 'distance_zero'.
 * If 'distance_zero' is true in state 7 it means that a defect position has
 * been reached, in state it denotes the end of the map.
 */

   always @(state or map or sof or distance_zero)
         case (state)
             0: nextstate <= map ? 1 : 0;              // state 0: idle (if no map)
             1: nextstate <= sof ? 3 : 1;              // state 1: idle until sof
             3: nextstate <= 2;                        // state 3: 1st init
             2: nextstate <= 6;                        // state 2: 2nd init
             6: nextstate <= 7;                        // state 6: 1st wait for defect
             7: nextstate <= distance_zero ? 5 : 6;    // state 7: 2nd wait for defect
             5: nextstate <= 4;                        // state 5: 1st correct defect
             4: nextstate <= distance_zero ? 13 : 12;  // state 4: 2nd correct defect
            12: nextstate <= 14;                       // state 12: 1st wait for defect
            14: nextstate <= distance_zero ? 5 : 6;    // state 14: 2nd wait for defect
            13: nextstate <= 15;
            15: nextstate <= 0;
            default: nextstate <= 0;
         endcase

   always @(posedge vp_clk or negedge resetb)
      if (~resetb) state <= 0;
      else state <= nextstate;


/* Generation, gating and re-timing of the 'correct' output
 *
 * 'correct' and 'filter code' are the two ouputs of the defect map navigator.
 * The combination of correct and filter_code define the operation of the
 * defect disguise input processor of SGSs main processor.
 * correct goes high every time a pixel is the data stream is to be replaced.
 * This is equivalent every time the down-counter counts down to zero, except
 * when it was counting as part of a 'jump' to a defect ('jumps' allow the
 * correction of defects further apart than the size of the defect distance
 * code permits).
 */
  
   assign apply_filter = ((state == 5) | (state == 4) | (state == 12) | (state == 14)
                          | (state == 13) | (state == 15));

   always @(posedge vp_clk or negedge resetb)
     if (~resetb) 
       correct <= 0;
     else 
       if (data_valid_posen)
	 correct <= !jumpd2 & apply_filter;

/* RAM address generator and chip enable
 *
 * Generates all RAM inputs necessary to read the defect codes.
 * Entering state 1 the address is reset, and each time the idle
 * state cycle is entered (ie. going into state 6 from anything
 * but state 7) the address is incremented (covering the normal
 * increment and first one during intialisation).
 */

   always @(posedge vp_clk or negedge resetb)
      if (~resetb) addr_event <= 0; 
      else addr_event <= addr_event_pre;
 
   always @(nextstate)
      addr_event_pre = ((nextstate == 1) | (nextstate == 12) | (nextstate == 14)
      | (nextstate == 6) | (nextstate == 7)); 
 
   always @(addr_event_pre or addr_event)
      addr_event_posen <= (addr_event_pre & !addr_event);

   always @(posedge vp_clk or negedge resetb)
      if (~resetb) ram_raddress <= 0;
      else if (addr_event_posen) 
         if ((nextstate == 0) | (nextstate == 1)) ram_raddress <= 0;
         else ram_raddress <= ram_raddress + 1;
      else ram_raddress <= ram_raddress; 

   assign ram_rceb = !((state == 3) | (state == 5));


/* Down-counter
 * 
 * Counts to the position of each defect by decrementing the distance portion
 * of each defect code. The clock 'data_valid' is used to clock the decrements
 * to give one per pixel in sync with the timing of the physical array readout.
 */

   always @(posedge vp_clk or negedge resetb)
      if (~resetb) distance_event <= 0; 
      else distance_event <= distance_event_pre;
 
   always @(nextstate or data_valid_pre)
      distance_event_pre <= (!(nextstate == 0) & !(nextstate == 1) & data_valid_pre)
      | (nextstate == 2) | (nextstate == 4); 
 
   always @(distance_event_pre or distance_event)
      distance_event_posen <= (distance_event_pre & !distance_event);
 
   assign distance_load_enable = ((state == 2) | (state == 3) | (state == 4) | (state == 5));

   always @(posedge vp_clk or negedge resetb) begin
      if (~resetb) begin
         distance_left <= ~0;
         filter_codei <= 0; 
         filter_code <= 0; 
         jump <= 0; 
         jumpd <= 0; 
         jumpd2 <= 0; 
      end
   else if (!distance_event_posen)
      begin
         distance_left <= distance_left;
         filter_code   <= filter_code;
         filter_codei  <= filter_codei;
         jump          <= jump;
         jumpd         <= jumpd;
         jumpd2        <= jumpd2;
      end
   else if (distance_load_enable) begin
         distance_left <= ram_data[15:2];
         filter_code <= filter_codei;
         filter_codei <= ram_data[1:0];
         if (ram_data[15:2] == 14'h3fff) jump <= 1;
         else jump <= 0;
      end
      else begin
         distance_left <= distance_left - 1;
         filter_code <= filter_codei;
         jumpd <= jump; 
         jumpd2 <= jumpd; 
      end
   end
 
   assign distance_zero = (distance_left == 0);


endmodule // of dc652a
