#!/bin/bash
# Colors for output
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Check if student ID is provided
if [ $# -ne 1 ]; then
    echo "Usage: $0 <student_ID>"
    exit 1
fi

STUDENT_ID=$1
STUDENT_TAR="${STUDENT_ID}.tar.gz"

# Check if the tar file exists
if [ ! -f "$STUDENT_TAR" ]; then
    echo -e "${RED}Error:${NC} File $STUDENT_TAR not found"
    exit 1
fi

# If a directory with the same name exists, remove it first
if [ -d "$STUDENT_ID" ]; then
    echo -e "${YELLOW}Removing existing directory $STUDENT_ID ...${NC}"
    rm -rf "$STUDENT_ID"
fi

echo -e "${YELLOW}Starting tests for student $STUDENT_ID...${NC}"

# Recreate student directory and extract
mkdir "$STUDENT_ID"
tar -xzf "$STUDENT_TAR" -C "$STUDENT_ID"

# Change into student directory
cd "$STUDENT_ID" || exit 1

# Define path to ABC executable (in student directory's external folder)
ABC="./external/abc"

# Create output and log directories inside student directory
mkdir -p output log

# Create external directory and symlink ABC
mkdir -p external
ln -s ../../external/abc ./external/abc

# Compile student code
echo -e "${YELLOW}Compiling student code...${NC}"
make
if [ ! -f "map" ]; then
    echo -e "${RED}FAILED${NC} (compile error)"
    exit 1
fi

# Already in student directory; begin testing
K_VALUES=(2 6 10)

TOTAL_TESTS=0
PASSED_TESTS=0

# Initialize TSV header and cost rows
HEADER_ROW=""
COST_ROW=""

echo "=================================="
for k in "${K_VALUES[@]}"; do
    echo -e "${YELLOW}Testing with k = $k${NC}"
    echo "----------------------------------"

    # Iterate over test_cases (located in test_cases folder one level up from script)
    for test_file in ../test_cases/*.blif; do
        base_name=$(basename "$test_file")
        test_name="${base_name%.blif}"
        output_file="output/k${k}_${base_name}"
        log_file="log/${test_name}.log"

        # Build TSV header row
        if [ -n "$HEADER_ROW" ]; then
            HEADER_ROW="${HEADER_ROW}\t"
            COST_ROW="${COST_ROW}\t"
        fi
        HEADER_ROW="${HEADER_ROW}${test_name}(k=$k)"

        echo -n "Testing $base_name: "
        TOTAL_TESTS=$((TOTAL_TESTS + 1))

        # Run map, executable is in current directory
        timeout 60s ./map -k "$k" "$test_file" "$output_file" > "$log_file" 2>&1
        code=$?
        if [ $code -eq 124 ]; then
            echo -ne "(timeout) "
        elif [ $code -ne 0 ]; then
            echo -ne "    (runtime error) "
        fi

        # Check if output file was generated
        if [ ! -f "$output_file" ]; then
            echo -e "${RED}FAILED${NC} (output file not generated)"
            COST_ROW="${COST_ROW}-"
            continue
        fi

        # Use ABC to check equivalence first
        abc_output=$($ABC -c "dcec $output_file $test_file" 2>&1)
        if ! echo "$abc_output" | grep -q "Networks are equivalent."; then
            echo -e "${RED}FAILED${NC} (networks not equivalent)"
            COST_ROW="${COST_ROW}-"
            continue
        fi

        # Extract the maximum K value from the output file
        extract_max_k() {
            local blif_file="$1"
            grep "^\.names " "$blif_file" | awk '{print NF-2}' | sort -n | tail -1
        }
        actual_max_k=$(extract_max_k "$output_file")

        # Check K constraint after equivalence is confirmed
        if [ $actual_max_k -gt $k ]; then
            echo -e "${RED}FAILED${NC} (K constraint violated: found K=$actual_max_k > required K=$k)"
            COST_ROW="${COST_ROW}-"
            continue
        fi

        # Both equivalence and K constraint passed - obtain statistics
        stats_output=$($ABC -c "read_blif $output_file; print_stats" 2>&1)

        lut_count=$(echo "$stats_output" | grep -o "nd *= *[0-9]*" | awk -F= '{print $2}' | sed 's/^[ \t]*//' | tail -1)
        circuit_level=$(echo "$stats_output" | grep -o "lev *= *[0-9]*" | awk -F= '{print $2}' | sed 's/^[ \t]*//' | tail -1)

        [ -z "$lut_count" ] && lut_count="-"
        [ -z "$circuit_level" ] && circuit_level="-"

        if [ "$lut_count" != "-" ] && [ "$circuit_level" != "-" ]; then
            cost=$((circuit_level * lut_count))
        else
            cost="-"
        fi

        echo -e "${GREEN}PASSED${NC}    |LUTs| = $lut_count, Level = $circuit_level, Cost = Level*|LUTs| = $cost"
        PASSED_TESTS=$((PASSED_TESTS + 1))
        COST_ROW="${COST_ROW}$cost"
    done

    echo "----------------------------------"
done

# Write TSV to result.tsv
echo -e "$HEADER_ROW" > result.tsv
echo -e "$COST_ROW" >> result.tsv

# Test summary
echo "=================================="
echo -e "${YELLOW}Test Summary:${NC}"
echo "Total tests: $TOTAL_TESTS"
echo "Passed tests: $PASSED_TESTS"
echo "Failed tests: $((TOTAL_TESTS - PASSED_TESTS))"
if [ "$PASSED_TESTS" -eq "$TOTAL_TESTS" ]; then
    echo -e "${GREEN}All tests passed!${NC}"
    exit_code=0
else
    echo -e "${RED}Some tests failed.${NC}"
    exit_code=1
fi

# Change back to parent directory
cd ..

exit "$exit_code"