#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

/*
 * The algorithm for each (plain-text, cipher-text) pair is
 * as follows.  Values for k (the period length) are considered in
 * turn from 1 up to the text length.  For each value of k:
 * 
 * 1. exists_permute is called to check that k is possible.  For every
 *    full block, it is checked that the frequencies of plain text characters
 *    matches that of the cipher text characters (if not one can't be a
 *    permutation of the other).  If the final block is not a full block
 *    then it isn't checked.  
 *
 * 2. The overall set of bit maps is initialised---one for each position
 *    in the line (0..linelen).  For overall[i], bit j is set if character
 *    j in the cipher text is the same as character i in the plain text, and is
 *    0 otherwise.
 *
 * 3. The k bitmaps (maps) are calculated based on the overall bitmaps.
 *    maps[i] shows for position i in the block, the set of bit positions
 *    in the cipher text that position i could be mapped to.  If any maps[i]
 *    is 0 then the current value of k must be wrong.
 *
 * 4. Try to find a permutation based on maps.  Start with the plaintext
 *    characters with the fewest possible cipher text positions.  Try all
 *    possible combinations.
 *  
 * 5. If a permutation is found print the details.
 */

/* Maximum length of cipher text and plain text strings handled. */
#define MAXLEN 256

/* Length of char array needed to hold a bitmap of MAXLEN bits. */
#define BIT_LEN (MAXLEN / 8)

typedef unsigned char *bitmap;

struct list {
    int posn;
    bitmap b;
    int num_bits;
    struct list *next;
} list_arr[MAXLEN], *head;

unsigned char plain[MAXLEN];
unsigned char cipher[MAXLEN];
unsigned char cipher2[MAXLEN*2];
int mapping[MAXLEN];
int linelen;
bitmap *maps;
bitmap *overall;

    static void
clear_bits(bitmap b)
{
    int i;

    for (i = 0; i < BIT_LEN; i++) {
	b[i] = 0;
    }
}

    static void
set_bits(bitmap b)
{
    int i;

    for (i = 0; i < BIT_LEN; i++) {
	b[i] = 0xff;
    }
}


    static int
get_bit(bitmap b, int bit)
{
    int word = bit / 8;
    int offset = bit % 8;

    return (b[word] >> offset) & 1;
}


    static void
set_bit(bitmap b, int bit, int value)
{
    int word = bit / 8;
    int offset = bit % 8;

    if (value) {
	b[word] |= 1 << offset;
    } else {
	b[word] &= ~(1 << offset);
    }
}


    static void
and_bits(bitmap op1, bitmap op2)
{
    int i;

    for (i = 0; i < BIT_LEN; i++) {
	op1[i] &= op2[i];
    }
}


    static int
zero_bits(bitmap b)
{
    int i;

    for (i = 0; i < BIT_LEN; i++) {
	if (b[i]) {
	    return 0;
	}
    }
    return 1;
}


    static int
count_bits(bitmap b)
{
    int i;
    int cnt = 0;

    for (i = 0; i < MAXLEN; i++) {
	if (get_bit(b, i)) {
	    cnt++;
	}
    }
    return cnt;
}


    static int
getline(FILE *in_file, unsigned char *buffer, int len)
{
    int lenread;

    if (fgets(buffer, len, in_file) == 0) {
	return 0;
    }
    lenread = strlen(buffer);
    if (lenread == 0 || (lenread == 2 && *buffer == '#')) {
	return 0;
    }
    if (buffer[lenread - 1] == '\n') {
	buffer[lenread - 1] = '\0';
    }
    return 1;
}

#define NUM_CHARS 256

    static int
exists_permute(int k, int linelen)
{
    int pcnt[NUM_CHARS], ccnt[NUM_CHARS];
    int i, block;

    for (block = 0; block + k < linelen; block += k) {
	for (i = 0; i < NUM_CHARS; i++) {
	    pcnt[i] = ccnt[i] = 0;
	}
	for (i = 0; i < k && block + i < linelen; i++) {
	    pcnt[plain[block + i]]++;
	    ccnt[cipher[block + i]]++;
	}
	for (i = 0; i < NUM_CHARS; i++) {
	    if (pcnt[i] != ccnt[i]) {
		return 0;
	    }
	}
    }
    return 1;
}


    static int
compute_map(int i, int k, int linelen)
{
    int posn;
    static bitmap tmp;
    int j;
    int block;

    if (!tmp) tmp = malloc(BIT_LEN);

    /*
     * Computing position i for a cycle length of k.
     */
    set_bits(maps[i]);
    for (posn = i; posn < linelen; posn += k) {
	clear_bits(tmp);
	block = (posn / k) * k;
	for (j = block; j < block + k; j++) {
	    if (j >= linelen || get_bit(overall[posn], j)) {
/*printf("set_bit: map %d, posn %d, cipher %d\n", i, posn, j);*/
		set_bit(tmp, j - block, 1);
	    }
	}
	and_bits(maps[i], tmp);
    }
}


    static int
search(struct list *list, bitmap used, int k)
{
    int i;

    if (!list) {
	return 1;
    }
    for (i = 0; i < k; i++) {
	if (get_bit(list->b, i) && !get_bit(used, i)) {
	    mapping[list->posn] = i;
	    set_bit(used, i, 1);
	    if (search(list->next, used, k)) {
		return 1;
	    }
	    set_bit(used, i, 0);
	}
    }
    return 0;
}


    static int
check_permute(int k)
{
    int i;
    struct list **lp;
    static bitmap used = 0;
    
    head = 0;
    for (i = 0; i < k; i++) {
	list_arr[i].posn = i;
	list_arr[i].num_bits = count_bits(maps[i]);
	list_arr[i].b = maps[i];
	for (lp = &head; *lp && (*lp)->num_bits < list_arr[i].num_bits;
	     lp = &(*lp)->next) {
	    ;
	}
	list_arr[i].next = *lp;
	*lp = &list_arr[i];
    }
    if (!used) {
	used = malloc(BIT_LEN);
    }
    clear_bits(used);
    return search(head, used, k);
}

    static void
print_permute(int k) 
{
    int i;

    printf("%1d", k);
    for (i = 0; i < k; i++) {
	printf(" %1d->%1d", i + 1, mapping[i] + 1);
    }
    printf("\n");
}


    static void
decipher(char *ctext, int k)
{
    int clen, i;
    char *cpos;
    int reverse_map[MAXLEN];

    clen = strlen(ctext);
    while (clen % k != 0) {
	strcat(ctext, "?");
	clen++;
    }
    for (i = 0; i < k; i++) {
	reverse_map[i] = mapping[i];
    }

    for (cpos = ctext; cpos < ctext + clen; cpos += k) {
	for (i = 0; i < k; i++) {
	    putchar(cpos[reverse_map[i]]);
	}
    }
    putchar('\n');
}


main()
{
    int found; 
    int i, j, k;
    int abort;
    FILE *in_file;

/*    if ((in_file = fopen("PROBLEMA.DAT", "r")) == 0) {
	fprintf(stderr, "Failed to open input file\n");
	exit(1);
    }*/
    in_file = stdin;

    maps = malloc(MAXLEN * sizeof(*maps));
    overall = malloc(MAXLEN * sizeof(*maps));
    for (i = 0; i < MAXLEN; i++) {
	maps[i] = malloc(BIT_LEN);
	overall[i] = malloc(BIT_LEN);
    }

    while (getline(in_file, plain, sizeof(plain)) &&
	   getline(in_file, cipher, sizeof(cipher)) &&
	   getline(in_file, cipher2, sizeof(cipher2))) {
	assert(strlen(plain) == strlen(cipher));

	found = 0;
	linelen = strlen(plain);
	for (i = 0; i < linelen; i++) {
	    for (j = 0; j < linelen; j++) {
		set_bit(overall[i], j, plain[i] == cipher[j]);
	    }
	}

	for (k = 1; k <= linelen && !found; k++) {
	    if (exists_permute(k, linelen)) {
		abort = 0;
		for (i = 0; !abort && i < k; i++) {
		    compute_map(i, k, linelen);
		    if (zero_bits(maps[i])) {
			abort = 1;
			break;
		    }
		}
		
		if (!abort) {
/*printf("Calling check, k = %d\n", k);*/
		    if (found = check_permute(k)) {
			decipher(cipher2, k);
/*			print_permute(k);*/
		    }
/*printf("Check returned\n");*/
		}
	    }
	}
	if (!found) {
	    printf("%s\n", cipher2);
	}
    }
}
