#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <GL/glut.h>
#include "algebra3.h"
#include "io.h"

#define MAX_N_REF 512
#define M_PI 3.14159265358979323846

GLfloat perspective_mat[16], modelview_mat[16], temp[16];
GLfloat topViewPos[3]={0,0,1800}, topViewDir[3]={0, 0, -1};
GLfloat modelSize;

//-------------------------------
// Global variables used in the
// main() and message handlers.
//-------------------------------
vec3 axis(0, 1, 0);
int mouse_button, mouse_state, mouse_downX, mouse_downY;
int winMain;              // Window IDs created by GLUT
Camera currentView, lastView;
int animation=0, frame=0, fps=0;
mat3 P1;
vec3 C1;

struct {
  int width, height;    // size of main window
  int step;
} config = {320, 240, 1};

//-----------------------------
// The more meaningful globals
//-----------------------------

DepthImage loadedImage;
vec3 modelCenter;

static void walk_forward(int d)
{
  vec3 normal = currentView.a ^ currentView.b;
  normal.normalize();
  currentView.COP += normal * d;
}

static void walk_sideway(int d)
{
  vec3 A = currentView.a;
  A.normalize();
  currentView.COP += A * d;
}

static void turn(int angle)
{
  mat4 M = rotation3D(axis, angle);
  
  currentView.a = M * currentView.a;
  currentView.b = M * currentView.b;
  currentView.c = M * currentView.c;
}

static void stdin_camera()
{
  float dest_camera[12];
  FILE* fp=fopen("camera.dat", "r");
  
  fscanf(fp, "%f %f %f\n", &dest_camera[0], &dest_camera[1], &dest_camera[2]);
  fscanf(fp, "%f %f %f\n", &dest_camera[3], &dest_camera[4], &dest_camera[5]);
  fscanf(fp, "%f %f %f\n", &dest_camera[6], &dest_camera[7], &dest_camera[8]);
  fscanf(fp, "%f %f %f\n", &dest_camera[9], &dest_camera[10], &dest_camera[11]);
  currentView.fromArray(dest_camera);
}

static void reshape(int w, int h)
{
  int win = glutGetWindow();

  glViewport(0, 0, w, h);

  if (win == winMain) {
    config.width = w; config.height = h;
  }
}

static void mouse(int button, int state, int x, int y)
{
  mouse_button = button;
  mouse_state = state;
  switch (state) {
  case GLUT_UP:
    lastView = currentView;
    printf("last view stored\n");
    break;
  case GLUT_DOWN:
    mouse_downX = x;
	mouse_downY = y;
    break;
  }
}

void motion(int x, int y)
{
  static int inited = 0;
  float xf = (float) (x-mouse_downX)/config.width;
  float yf = (float) (y-mouse_downY)/config.height;
  float angle;
  mat4 M;
  vec3 normal, a;

  if (! inited) {
    lastView = currentView;
    inited = 1;
  }
  
  switch (mouse_button) {
  case GLUT_LEFT_BUTTON:  // rotate
    angle = -90*xf;
	axis = lastView.b;
    M = rotation3D(axis, angle);  // horizontal 
	angle = 90*yf;
	axis = lastView.a; 
	M = rotation3D(axis, angle) * M;  // vertical
    currentView.a = M * lastView.a;
    currentView.b = M * lastView.b;
    currentView.c = M * lastView.c;
	//M = translation3D(modelCenter)* M * translation3D(-modelCenter);
	currentView.COP = M * lastView.COP;
    glutPostRedisplay();
    break;
  case GLUT_MIDDLE_BUTTON:
    // nothing
    break;
  case GLUT_RIGHT_BUTTON:  // Walk
    a = lastView.a;
    a.normalize();
    normal = lastView.a ^ lastView.b;
    normal.normalize();
    currentView.COP = lastView.COP + normal * ((-yf)*config.step)
                                   + a * ((xf)*config.step);
    glutPostRedisplay();
    break;
  }
}

//========================================================================
// The following macro is used exclusively by the function display().
#define MACRO_Warp_Pixel \
			delta = loadedImage.getDisparity(x, y);	\
			if (delta < 0.000001) continue;  /* skip the background */	\
													\
			r2 = a*x + b*y + c + j*delta;			\
		    s2 = d*x + e*y + f + k*delta;			\
			t2 = g*x + h*y + i + l*delta;			\
													\
			if (t2 > 0) { /* not behind the eye */  \
			  rgb = loadedImage.getColor(x, y);		\
			  u2 = r2/t2;							\
			  v2 = s2/t2;							\
			  fb.writeColor(u2, v2, rgb);				\
			}
			  //glColor4ub(rgb[0], rgb[1], rgb[2], 255);	\
      	      glVertex3f(u2, v2, t2*0.0001);		\
			}

//========================================================================
void display(void)
{
  int x, y;
  vec3 eye, lookat, up, center;
  float scale;
  unsigned char* rgb;
  Camera cam = currentView;
  GLenum error;
  mat3 P2, P2inv, P1inv, M;
  vec3 C2, V, epipole;
  float a, b, c, d, e, f, g, h, i, j, k, l;
  float epX, epY;
  float delta, r2, s2, t2, u2, v2;
  FrameBuffer fb;

  //================================================
  // Update the Output Image Window
  //================================================

  // Adjust vectors a and b to match the output resolution.
  // The same scaling is done for both a and b to avoid distortion.
  scale = (float) loadedImage.width / config.width;
  cam.a *= scale;
  cam.b *= scale;

    /*-------- Set up the OpenGL matrices --------*/
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
	glTranslatef(-1, 1, 0);
	glScalef(2.0/config.width, -2.0/config.height, 1);
  
    glClearColor(0, 0, 1, 1);  // blue background
	glClearDepth(1.0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glDisable(GL_DEPTH_TEST);  // Don't need it if warping with occlusion-compatible order
  

	/* compute the constants used in the warping equation. */
    cam.toMat3(P2);
	P2inv = P2.inverse();
	C2 = cam.COP;

	M = P2inv * P1;
	a = M[0][0]; b = M[0][1]; c = M[0][2];
    d = M[1][0]; e = M[1][1]; f = M[1][2];
    g = M[2][0]; h = M[2][1]; i = M[2][2];

    V = P2inv * (C1 - C2);
    j = V[0]; k = V[1]; l = V[2];

	/* compute the epipole */
	P1inv = P1.inverse();
	epipole = P1inv * (C2 - C1);
	epX = (epipole[0] / epipole[2]);
	epY = (epipole[1] / epipole[2]);

	/* epipole at infinity? */
	epX = MAX(epX, -10000);  
	epY = MAX(epY, -10000);
	epX = MIN(epX, 10000);
	epY = MIN(epY, 10000);
	printf("epipole=%f, %f\n", epX, epY);

  /*-------- Do the actual rendering --------*/
	fb.setSize(config.width, config.height, 1);
//   glBegin(GL_POINTS);

	//=========================================================
	// Positive Epipole: warp in the directions toward epipole
	//=========================================================
	if (epipole[2] >= 0) {
	for (y=0; y<loadedImage.height && y<=epY; y++) {
	    for (x=0; x<loadedImage.width && x<=epX; x++) {

			MACRO_Warp_Pixel
		
		} // for x
	    for (x=loadedImage.width-1; x>=0 && x>epX; x--) {

			MACRO_Warp_Pixel
		
		} // for x
	} // for y

	for (y=loadedImage.height-1; y>=0 && y>epY; y--) {
	    for (x=0; x<loadedImage.width && x<=epX; x++) {

			MACRO_Warp_Pixel
		
		} // for x
	    for (x=loadedImage.width-1; x>=0 && x>epX; x--) {

			MACRO_Warp_Pixel
		
		} // for x
	} // for y
	} // positive epipole
	else {
	//============================================================
	// Negative Epipole: warp in the directions away from epipole
	//============================================================
	for (y=MIN(loadedImage.height-1, epY); y>=0; y--) {
	    for (x=MIN(loadedImage.width-1, epX); x>=0; x--) {

			MACRO_Warp_Pixel
		
		} // for x
	    for (x=MAX(0, epX+1); x<loadedImage.width; x++) {

			MACRO_Warp_Pixel
		
		} // for x
	} // for y

	for (y=MAX(0, epY+1); y<loadedImage.height; y++) {
	    for (x=MIN(loadedImage.width-1, epX); x>=0; x--) {

			MACRO_Warp_Pixel
		
		} // for x
	    for (x=MAX(0, epX+1); x<loadedImage.width; x++) {

			MACRO_Warp_Pixel
		
		} // for x
	} // for y
	}
	
//	glEnd();
	fb.flipY();
	fb.display();
//	glutSwapBuffers();

  fps++;

  /* Check if any error occurs. */
  while (GL_NO_ERROR != (error=glGetError())) {
	printf("%s\n", gluErrorString(error)); 
  }
}

static void key(unsigned char key, int x, int y)
{
  int i, j;
  GLfloat perspective_mat[16];

  switch(key) {
  case 'm':
    printf("Current Projection Matrix...\n");
    glMatrixMode(GL_PROJECTION);
    glGetFloatv(GL_PROJECTION_MATRIX, perspective_mat);
    for (i=0; i<4; i++) {
      for (j=0; j<4; j++)
	printf("%f ", perspective_mat[j*4+i]);
      printf("\n");
    }
    printf("\n");

    printf("camera setup...\n");
    printf("COP=(%f %f %f)\n",
	   currentView.COP[0], currentView.COP[1], currentView.COP[2]);
    printf("A=(%f %f %f)\n",
	   currentView.a[0], currentView.a[1], currentView.a[2]);
    printf("B=(%f %f %f)\n", 
	   currentView.b[0], currentView.b[1], currentView.b[2]);
    printf("C=(%f %f %f)\n", 
	   currentView.c[0], currentView.c[1], currentView.c[2]);
    printf("\n");
    break;
  case 'A': // change rotation axis
    float f1, f2, f3;
    printf("Enter new rotation axis: ");
    scanf("%f %f %f", &f1, &f2, &f3);
    axis = vec3(f1, f2, f3).normalize();
    printf("New axis = (%.3f %.3f %.3f).\n", axis[0], axis[1], axis[2]);
    break;
  case 'b':
    walk_forward(-config.step);
    glutPostRedisplay();
    break;
  case 'c':
    stdin_camera();
    axis = -currentView.b;
    axis.normalize();
    glutPostRedisplay();
    break;
  case 'f':
    walk_forward(config.step);
    glutPostRedisplay();
    break;
  case 'l':
    walk_sideway(-config.step);
    glutPostRedisplay();
    break;
  case 'r':
    walk_sideway(config.step);
    glutPostRedisplay();
    break;
  case 't':
    turn(config.step);
    glutPostRedisplay();
    break;
  case '+':
    config.step *= 2;
    printf("New step size = %d\n", config.step);
    break;
  case '-':
    config.step /= 2;
    if (config.step < 1) config.step = 1;
    printf("New step size = %d\n", config.step);
    break;
  case 'R':
    currentView = lastView = loadedImage.camera;
    glutPostRedisplay();
    break;
  case 'q':
  case '\033':
    exit(0);
  }
}

static void idle()
{
  if (animation) {
    glutSetWindow(winMain);
    glutPostRedisplay();
  }
  else {
    frame = 0;
  }
}

static void timer(int value)
{
	printf("%d fps\n", fps);
	fps=0;
    glutTimerFunc(1000, timer, 0);
}

static void setCallbacks(int win)
{
  glutSetWindow(win);

  //
  // Register GLUT callbacks
  //
  glutReshapeFunc(reshape);
  glutDisplayFunc(display);
  glutMotionFunc(motion);
  glutMouseFunc(mouse);
  glutKeyboardFunc(key);
  glutTimerFunc(1000, timer, 0);
}

static void computeCenter(DepthImage& dImg)
{
	int x, y, init=0;
	vec3 p, pMin, pMax;

	for (y=0; y<dImg.height; y++) {
		for (x=0; x<dImg.width; x++) {
			if (dImg.getDisparity(x, y)<0.000001) continue;
			p = dImg.location_R3(x, y);
			if (! init) {  // Initialize pMin and pMax
				init = 1;
				pMin = pMax = p;
			}
			else {
				pMin = min(p, pMin);
				pMax = max(p, pMax);
			}
		}
	}
	modelCenter = (pMin + pMax) / 2.0;
}

void main(int argc, char *argv[])
{
  //-----------------------------------------------------
  // Load the reference image and initialize the camera
  // to the one used by the loaded image.
  //-----------------------------------------------------
  loadedImage.loadRawFile("bunny.wrp");
  currentView = loadedImage.camera;
  currentView.toMat3(P1);
  C1 = currentView.COP;
  config.width  = loadedImage.width;
  config.height = loadedImage.height;

  glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH);
  glutInitWindowSize(config.width, config.height);
  glutInit(&argc, argv);

  winMain  = glutCreateWindow("Warper");
  glShadeModel(GL_FLAT);
  glClearColor(0.0, 0.0, 0.0, 1.0);
  glClearDepth(1.0);

  setCallbacks(winMain);

  // Query the machine support.  
  GLfloat v[2];
  printf("glIsEnabled(GL_POINT_SMOOTH) returns %d\n",
  		  glIsEnabled(GL_POINT_SMOOTH));
  glGetFloatv(GL_POINT_SIZE_RANGE, v);
  printf("glGet(GL_POINT_SIZE_RANGE) returns %f, %f\n", v[0], v[1]);
  glGetFloatv(GL_POINT_SIZE_GRANULARITY, v);
  printf("glGet(GL_POINT_SIZE_GRANULARITY) returns %f\n", v[0]);

  glutMainLoop();
}
