/*
 *  Description: Camera Head Serial Interface
 *
 *  Designer : Paul Likoudis
 *  Company  : VLSI Vision Ltd.
 *  Module   : si653a
 *
 * Revision History:
 * DD-MMM-YY:  Comment
 * 21-MAY-97:  Copy from CPiA original
 * 10-JUN-97:  cam_at/cr removed
*/

module si653a (
               mhz14,
               clrb,
               wr63,
               wrb,
               scl,
               sda,
	       sif,
               ad,
               din2,
               din0,
               all_x,
               fast_serial,
               acc_sda_scl,
               cam_tms,
               cam_setup0,
               cam_setup1,
               cam_setup2,
               cam_setup3,
               scl_sif,
               sda_sif,
               scl_,
               sda_,
               active,
               nak,
               sft_rst,
               ram_ceb,
               ram_address,
               ram_data,
               e2read,
               map
               );

   input mhz14;
   input clrb;
   input wr63;
   input wrb;
   input scl;
   input sda;
   input [1:0] sif;
   input [7:0] ad;
   input       din2;
   input       din0;
   input [20:0]  all_x;    //  mirrored regs (exp)

  input           fast_serial,
                  acc_sda_scl;

  input     [7:0] cam_tms;

  input     [7:0] cam_setup0,
                  cam_setup1,
                  cam_setup2,
                  cam_setup3;

  output          scl_sif,
                  sda_sif;

   output scl_;
   output sda_;
   output active;
   output nak;
   output [1:0] sft_rst;
   output ram_ceb;
   output [6:0] ram_address;
   output [15:0] ram_data;
   output e2read;
   output map;


   reg scl_;
   reg sda_;
   reg active;
   reg nak;
   reg [1:0] sft_rst;
   reg ram_ceb;
   reg [6:0] ram_address;
   reg [15:0] ram_data;
   reg e2read;
   reg map;

   reg mapi;
   reg [7:0] ix;
   reg [7:0] shifty;
   reg [7:0] shifter;
   reg [3:0] bit_count;

   reg iscl;            // iscl pulses: scl+S+P
   reg [4:0] oneoff;
   reg [6:0] ckdiv;
   reg [7:0] ad_;
  
   wire fast_ck  = fast_serial ? (ckdiv[3:0]=='hf) : (ckdiv[6:0]=='h7f);
   wire fast_ckb = fast_serial ? (ckdiv[3:0]=='h7) : (ckdiv[6:0]=='h3f);
   assign {sda_sif,scl_sif} = acc_sda_scl ? {sda,scl} : sif; 

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) sft_rst <= 1;
      else if (ad == 57 & ~wrb) sft_rst <= {din2, din2|din0};
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) ad_ <= 0;
      else if (~wrb & ~active) ad_ <= ad;
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) begin
          e2read <= 1;
          map <= 0;
      end
      else if (!active) begin
         e2read <= 0;
         map <= mapi;
      end
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) nak <= 0;
      else if (fast_ck && ~iscl)
        nak <= (bit_count == 9) && sda && oneoff != `OO_EOEE;
      else
        nak <= 0;
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) mapi <= 0;
      else if (bit_count == 9 && !scl_) begin
         case (oneoff)
            `OO_PFL1: mapi <= 1;
            default: mapi <= mapi;
         endcase
      end
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) begin
         oneoff <= `OO_INIT;
         ram_ceb <= 1;
      end
      else if (bit_count == 9 && !scl_ && fast_ck) begin
         case (oneoff)
          `OO_PFHI:
	      if (shifty == 'h00)
		oneoff <= sda ? `OO_NOEE : oneoff + 1;
	      else
		oneoff <= `OO_PFL1;
	  `OO_PFL2:
	     if (shifty == 'h00)
	       oneoff <= sda ? `OO_NOEE : oneoff + 1;
	     else
	       oneoff <= `OO_PFHI;
	  `OO_PFL1: oneoff <= `OO_PFHI;
	  `OO_EODM: oneoff <= sda ? `OO_NOEE : oneoff + 1;
	  `OO_WTD0: oneoff <= (ix != 'ha0 || sda) ? `OO_NOEE : oneoff + 1;
	  `OO_WTD4: oneoff <= `OO_NOEE;
	   default: oneoff <= sda ? `OO_NOEE : oneoff + 1;
	 endcase // case (oneoff)
	ram_ceb <= !(oneoff==`OO_PFHI || oneoff==`OO_EODM);
      end
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) begin
         ram_address <= 127;
         ram_data <= 0;
      end
      else if (bit_count == 9 && fast_ck && ~iscl) begin
         case (oneoff)
            `OO_PFL2,
	    `OO_PFL1: begin
	       ram_address <= ram_address + 1;
	       ram_data <= {shifter[7:0],shifty};
	    end
	 endcase
      end
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) ix <= 0;
      else if (wr63) begin
         case (ad)
            63: ix <= 'ha0; // fineH (inc)
            47: ix <= 'h74; // tms
            57: ix <= 'h10; // su0
            58: ix <= 'h11; // su1
            59: ix <= 'h14; // su4
            60: ix <= 'h15; // su5
            default: ix <= 0;
         endcase
      end
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) active <= 1;
      else if (wr63) active <= 1;
      else if (bit_count == 15 & fast_ck & ~iscl)
         active <= oneoff !=`OO_NOEE ? active : 0;
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) bit_count <= ~0;
      else if (~active) bit_count <= ~0;
      else  if (iscl & fast_ck) begin
         case (bit_count)
            9: begin
                  if (oneoff == `OO_NOEE) bit_count <= 10; // just as if no eeprom there
                  else if (oneoff == `OO_WRA1 || oneoff == `OO_WTS2) bit_count <= ~0;
                  else bit_count <= active ? 1 : ~0;
               end
           10: bit_count <= ~0; // stop after def_map
           default: bit_count <= bit_count + 1;
         endcase
      end
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) {shifty, shifter} <= 0;
      else if (~active) {shifty, shifter} <= 0;
      else if (~iscl & fast_ck) begin
         shifter <= {shifter, shifty[7]};
         casex (bit_count)
            0: case (oneoff)			 // start condition
                  `OO_INIT: shifty <= 'h20;	 // init sensor
                  `OO_WTS2: shifty <= 'ha0;	 // init eeprom
                  `OO_WRA1: shifty <= 'ha1;	 // init eeprom
                  default: shifty <= 'h20;	 // write to sensor
               endcase
            9: case (oneoff)
                  `OO_INIT: shifty <= 'h10;	 // cam_setup ix
                  `OO_WTS0: shifty <= 4;	 // soft reset
                  `OO_WTS2: shifty <= 32;	 // eeprom start
                  `OO_EOEE: shifty <= shifty;
                  `OO_NOEE: shifty <= ix;
                  `OO_WTIX: shifty <= dout(ad_,
                                           all_x,
                                           cam_tms,
                                           cam_setup0,
                                           cam_setup1,
                                           cam_setup2,
                                           cam_setup3);
                  `OO_WTD0,
                  `OO_WTD1,
                  `OO_WTD2,
                  `OO_WTD3,
                  `OO_WTD4: shifty <= dout(oneoff[2:0],
                                           all_x, 
                                           cam_tms,
                                           cam_setup0,
                                           cam_setup1,
                                           cam_setup2,
                                           cam_setup3);
	       endcase
            default: shifty <= {shifty, sda};
         endcase
      end
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) {scl_, iscl} <= ~0;
      else if (acc_sda_scl) {scl_, iscl} <= {2{sif[0]}};
      else if (~active) {scl_, iscl} <= ~0;
      else if (fast_ck) begin
         iscl <= ~iscl;
         if ((oneoff == `OO_NOEE && bit_count == 10) || (&bit_count)) scl_ <= 1;
         else scl_ <= ~iscl;
      end
   end

   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) sda_ <= 1;
      else if (acc_sda_scl) sda_ <= sif[1];
      else if (~active) sda_ <= 1;
      else if (~iscl & fast_ckb) begin
         if ((oneoff == `OO_NOEE || oneoff == `OO_WRA1 || oneoff == `OO_WTS2) &&
             (bit_count == 10 || bit_count == 0 || bit_count == 15))
            sda_ <= bit_count == 15;  // stop/start EEPROM or sensor
         else if (oneoff > `OO_WRA1 && oneoff < `OO_NOEE
                  || oneoff > `OO_WTD4 && oneoff !=`OO_INIT) // ack. EEPROM
            sda_ <= !(bit_count == 9 && !(oneoff == `OO_EOEE && shifty == 'h00));
         else  // write to sensor or EEPROM
            sda_ <= bit_count == 9 ? 1 : shifty[7];
      end
   end

   function [7:0] dout;
      input [5:0] i;
//      input [599:0] all_buns;
  input    [20:0] all_x;

  input     [7:0] cam_tms;         //equiv all_buns 432:425

  input     [7:0] cam_setup0,      //equiv all_buns 512:505
                  cam_setup1,      //equiv all_buns 520:513
                  cam_setup2,      //equiv all_buns 528:521
                  cam_setup3;      //equiv all_buns 536:529


      begin
         case (i)
             2:  dout = all_x[0];
             3:  dout = all_x[8:1];
             4:  dout = all_x[9];
             5:  dout = all_x[17:10];
             6:  dout = all_x[20:18];
            47:  dout = cam_tms;
            57:  dout = cam_setup0;
            58:  dout = cam_setup1;
            59:  dout = cam_setup2;
            60:  dout = cam_setup3;
            default: dout = all_x[0];
         endcase
      end
   endfunction


   always @ (posedge mhz14 or negedge clrb) begin
      if (~clrb) ckdiv <= 0;
      else ckdiv <= ckdiv + 1;
   end


endmodule //of si652a
