/*
 *  Description: RGB matrix & Ap_cor
 *
 *  Designer : Stewart G. Smith
 *  Company  : VLSI Vision Ltd.
 *  Module   : matrix
 *
 * Revision History:
 * DD-MMM-YY:  Comment
 * 21-MAY-97:  Copy from CPiA original
*/



/*************************************************
matrix & apcor
Note interpretation of coeffs: on diag c =  2(1 + c'), off diag c = 2c'.
4 passes, RGB and X, in X the next apcor val is computed and stored
rhot is pclkd2
*************************************************/
module matrix (insig, resetb, vp_clk, cof00, cof01, cof02, cof10, cof11, cof12,
		cof20, cof21, cof22, ushpin, oenv, apcor, apthresh,
		rhot, pclkd1, xhot, out, oenvd2, oenvd6);
	parameter DBITS8 = 9;
	parameter CBITS8 = 8;
	parameter OBITS8 = 13;
	parameter RBITPOSN8 = (DBITS8+CBITS8) - (OBITS8+1);

	input [DBITS8-2:0] insig;
	input [CBITS8-1:0] cof00, cof01, cof02, cof10, cof11, cof12, cof20, cof21, cof22;
	input [11:0] ushpin;
	input [3:0] apcor, apthresh;
	input vp_clk, resetb, oenv, rhot, pclkd1, xhot;
	output [8:0] out;
	output oenvd2, oenvd6;
	reg [8:0] out;
	wire [8:0] outsel;
	reg [8:0] ushpcored;
	wire [7:0] apcorint;
	wire [8:0] ushpclip;
	wire [9:0] apthreshint;
	reg [9:0] gthri;
	reg [8:0] gthr;
	reg [8:0] rline, gline, bline, rlinei, glinei, blinei;
	reg [CBITS8-1:0] rcofsel, gcofsel, bcofsel;
	wire [DBITS8-1:0] rlinese, glinese, blinese, rlinechoice;
	wire [OBITS8-1:0] rlineprodi, glineprodi, blineprodi;
	wire [OBITS8:0] rlineprod, glineprod, blineprod;
	reg [12:0] ushpcatch;
	wire [OBITS8+1:0] gbprodsum;
	wire [OBITS8+2:0] rgbprodsum;
	wire [OBITS8+2:0] apcorsum;
	reg ghot, oenvd2, oenvd6;
	wire greyback, rgdatapassing;

// generate ghot locally for synth/loading reasons
	always @ (posedge vp_clk or negedge resetb) begin
		if(!resetb)
			ghot <= 0;
		else
			ghot <= rhot;
	end
	
//	assign apcorint = {1'd0,apcor,3'd0};
	assign apcorint = {2'd0,apcor,2'd0};
	assign apthreshint = {3'd0,apthresh,3'd0};
	assign greyback = &apcor;

// envelope and clip 12-bit signed unsharp signal to 9-bit signed
	function [8:0] biclip9;
		input[11:0] in;
		case(in[11])
			0:	case(in[10] | in[9] | in[8])
 					0:	biclip9 = in[8:0];
 					1:	biclip9 = 9'd255;
				endcase
			1:	case(in[10] & in[9] & in[8])
 					1:	biclip9 = in[8:0];
 					0:	biclip9 = 9'd256;
				endcase
		endcase
	endfunction
	assign ushpclip = oenv ? biclip9(ushpin) : 0;

// core the unsharp signal
	always @ (ushpclip or apthreshint) begin
		case(ushpclip[8])
			0:      gthri = {ushpclip[8],ushpclip,1'b1} + {~apthreshint,1'b1};
			1:      gthri = {ushpclip[8],ushpclip,1'b0} + {apthreshint,1'b0};
		endcase
		gthr = gthri[9:1];
		case(ushpclip[8] ^ gthr[8])
			0:      ushpcored = gthr;
			1:      ushpcored = 0;
		endcase
	end

// rhot means Red at module input
	assign rgdatapassing = rhot | ghot;

// envelope and sign extend caught colors
	assign rlinese = oenvd2 ? {1'b0,rline} : 0;
	assign glinese = oenvd2 ? {1'b0,gline} : 0;
	assign blinese = oenvd2 ? {1'b0,bline} : 0;

// truncating multiplier, bipolar data capability for ushp signal
	function [OBITS8-1:0] tm090813;
		input [DBITS8-1:0] dat;
		input [CBITS8-1:0] cof;
		reg [DBITS8+CBITS8-1:0] fullprod;
		reg [DBITS8:0] hiprod;
		reg [CBITS8-1:0] loprod;
		reg [CBITS8-3:0] lotmp;
		reg [CBITS8:0] mbcof;
		reg [DBITS8:0] pp, pp1;
		reg [DBITS8:0] ppsds2;
		reg [DBITS8+3:0] pps;

		integer i;

		begin
			mbcof = {cof,1'b0};
			ppsds2 = 0;
			lotmp = 0;
			for(i=2; i<=CBITS8; i = i + 2) begin
				case(mbcof[2:0])
					0:      pp1 = 0;
					1:      pp1 = {dat[DBITS8-1],dat};
					2:      pp1 = {dat[DBITS8-1],dat};
					3:      pp1 = {dat,1'b0};
					4:      pp1 = {dat,1'b0};
					5:      pp1 = {dat[DBITS8-1],dat};
					6:      pp1 = {dat[DBITS8-1],dat};
					7:      pp1 = 0;
				endcase
				case(mbcof[2])
					0:      pp = pp1;
					1:      pp = ~pp1;
				endcase
				pps = {ppsds2[DBITS8],ppsds2[DBITS8],ppsds2,mbcof[2]}
					+ {pp[DBITS8],pp[DBITS8],pp,mbcof[2]};
				ppsds2 = {pps[DBITS8+3:3]};
				loprod = {pps[2:1],lotmp};
				mbcof = mbcof >> 2;
				lotmp = loprod >> 2;
			end
			hiprod = pps[DBITS8+3:3];
			fullprod = {hiprod,loprod};
			tm090813 = fullprod[DBITS8+CBITS8-1:DBITS8+CBITS8-OBITS8];
		end

	endfunction

// coefficient selection (pclkd1 at module i/p means output r hot here)
// want oenv2 in a casex but doesn't seem to work, hence ?: construct
	always @ (cof00 or cof01 or cof02 or cof10 or cof11 or cof12 or cof20 or cof21
			or cof22 or apcorint or oenvd2 or rhot or xhot or pclkd1 or ghot) begin
		casex({pclkd1,xhot,rhot,ghot})
		4'b1000:	begin
						rcofsel = oenvd2 ? cof00 : apcorint;
						gcofsel = oenvd2 ? cof01 : apcorint;
						bcofsel = oenvd2 ? cof02 : apcorint;
					end
		4'b0100:	begin
						rcofsel = oenvd2 ? cof10 : apcorint;
						gcofsel = oenvd2 ? cof11 : apcorint;
						bcofsel = oenvd2 ? cof12 : apcorint;
					end
		4'b0010:	begin
						rcofsel = oenvd2 ? cof20 : apcorint;
						gcofsel = cof21;
						bcofsel = cof22;
					end
		   default: begin
						rcofsel = apcorint;
						gcofsel = cof21;
						bcofsel = cof22;
					end
		endcase
	end

// data selection
	always @ (insig or rline or gline or blinei or rgdatapassing) begin
		case(rgdatapassing)
			0: begin
					rlinei = rline;
					glinei = gline;
					bline = insig;
				end
			1:	begin
					rlinei = gline;
					glinei = insig;
					bline = blinei;
				end
		endcase
	end


// 3 multiplier instances (R does unsharp too)
	assign rlinechoice = ghot ? ushpcored : rlinese;
	assign rlineprodi = tm090813(rlinechoice,rcofsel);
	assign glineprodi = tm090813(glinese,gcofsel);
	assign blineprodi = tm090813(blinese,bcofsel);

// interpret on-diagonal coefficients as 1 + c, off-diagonal as c
// again pclkd1 at module i/p means output r hot here
// stick on blue
// use oenvd2 to stop rounding bit waggling in blanking time
// round on the on-diag sum, for later 2-bit downshift after apcor
	assign rlineprod = pclkd1 ? {2'd0,rlinese,2'd0,oenvd2} + {rlineprodi[12],rlineprodi} :
		{rlineprod[12],rlineprodi};
	assign glineprod = xhot ? {2'd0,glinese,2'd0,oenvd2} + {glineprodi[12],glineprodi} :
		{glineprod[12],glineprodi};
	assign blineprod = rgdatapassing ? {2'd0,blinese,2'd0,oenvd2} + {blineprodi[12],blineprodi} :
		{blineprod[12],blineprodi};

// sum the 3 products
	assign gbprodsum = {glineprod[13],glineprod} + {blineprod[13],blineprod};
	assign rgbprodsum = greyback ? 256 :
					{rlineprod[13],rlineprod[13],rlineprod} + {gbprodsum[14],gbprodsum};

// apcor sum
//	assign apcorsum = rgbprodsum[15:1] + {{2{ushpcatch[12]}},ushpcatch};
	assign apcorsum = rgbprodsum + {{3{ushpcatch[12]}},ushpcatch};

// clip apcor'd RGB to 9-bit unsigned, with 1-bit downshift (rounded earlier)
	function [11:0] hiclip9;
		input[15:0] in;
		case(in[15])
			0:	begin
				case(in[14] | in[13] | in[12] | in[11] | in[10])
 					0:	hiclip9 = in[9:1];
 					1:	hiclip9 = 9'd511;
				endcase
				end
			1:	hiclip9 = 9'd0;
		endcase
	endfunction

	assign outsel = ghot ? out : hiclip9(apcorsum);

// RGB color catcher and output clock
	always @ (posedge vp_clk or negedge resetb) begin
		if(!resetb) begin
			rline <= 0;
			gline <= 0;
			blinei <= 0;
			out <= 0;
		end else begin
			rline <= rlinei;
			gline <= glinei;
			blinei <= bline;
			out <= outsel;
		end
	end

// ushp catcher (full pixel period)
	always @ (posedge vp_clk or negedge resetb) begin
		if(!resetb) begin
			ushpcatch <= 0;
		end else begin
			ushpcatch <= ghot ? rlineprodi[12:0] : ushpcatch;
		end
	end

// retime oenv for delays of 2 (local & coder420 clear) and 6 (cofspace cofsel env)
	always @ (posedge vp_clk or negedge resetb) begin
		if(!resetb) begin
			oenvd6 <= 0;
			oenvd2 <= 0;
		end else begin
			oenvd6 <= ghot ? oenvd2 : oenvd6;
			oenvd2 <= ghot ? oenv : oenvd2;
		end
	end

endmodule
