/*
 *  colctrl.v
 *
 *  Copyright VLSI Vision Ltd
 *
 *  Designer: Stewart Smith & Andrew Murray
 *  Module:   Colour balance engine
 *
 * Revision History:
 * DD-MMM-YY:  Comment
 * 05-JUN-97   Original
 * 05-JUN-97   Channel sums reset to 111....111 in remote mode (was 000....000)
 * 09-JUN-97   Takes 12-bit stats now (was 8-bit)
 * 10-JUN-97   fixed NOHUNTVAL replaced by proportional (racc >> 6)
 * 27-AUG-97:  mu extended to 7 bits for testability (PL)
 *
 */


`define ROUNDINGBIT (1 << (`BOTBIT - 1))
`define TOPBIT 14
`define BOTBIT 7
`define BOTBITP4 11
`define BITPLANES 8
`define ACBALLONES ((1 << (`TOPBIT + 2)) - 1)
/* `define MAXPOS ((1 << (`TOPBIT + 1)) - 1) */
`define MAXPOS 127
`define MAXNEG 128
// `define NOHUNTVAL 3
`define NOHUNTVAL 48

// DEV MOD 1 of 4: SHORT NAME FOR DEV
module colctrl (clk, resetb, run, racc, gacc, bacc, remote, freeze, pregain,
	rinr, ginr, binr,
	ring, ging, bing,
	rinb, ginb, binb,
	gnbase, mu,
// DEV MOD 2 of 4: EXTRA EXTERNALS
/*
    subtract_ff, op1_ff, op2_ff,
    state,
    channel, cow,
    result,
    stable, r_ismax, r_ismin, g_grtrthn_b,
    ronbase, gonbase, bonbase, rclamp, gclamp, bclamp, rpeg, gpeg, bpeg, normal, pullback,
*/
	m_rinr, m_ginr, m_binr,
	m_ring, m_ging, m_bing,
	m_rinb, m_ginb, m_binb);

  /* system level inputs */
  input resetb, clk;
  input run;    // frame-level timing control
//   input [7:0] racc, gacc, bacc;   // above-threshold energy accumulations
  input [11:0] racc, gacc, bacc;   // above-threshold energy accumulations
  /* user inputs */
  input remote, freeze, pregain;   // mode controls
  input [7:0] rinr, ginr, binr;
  input [7:0] ring, ging, bing;    // input matrix
  input [7:0] rinb, ginb, binb;
  input [3:0] gnbase;       // target for the lowest channel gain
  input [7:0] mu;           // balance time constant
  /* outputs */
  output [7:0] m_rinr, m_ginr, m_binr;
  output [7:0] m_ring, m_ging, m_bing;   // output matrix
  output [7:0] m_rinb, m_ginb, m_binb;
  reg [7:0] m_rinr, m_ginr, m_binr;
  reg [7:0] m_ring, m_ging, m_bing;   // output matrix
  reg [7:0] m_rinb, m_ginb, m_binb;
  /* internals */
  reg [7:0] ielement;        // value of current matrix element
  reg [4:0] state;        // state machine state
  reg [1:0] channel;      // 'hot' colour channel
  reg [1:0] cow;          // 'hot' matrix column or row
  reg [3:0] bitplane;     // 'hot' cycle of the shift-add multiply
  reg [`TOPBIT+1:0] gain_plus_gnbase; // temp. storage
  reg [`TOPBIT+1:0] rsum, gsum, bsum;  // long term error accumulators
  reg [`TOPBIT+1:0] op1, op2, result;
  reg [7:0] outword;
  reg subtract; // ALU control variable
  reg ronbase, gonbase, bonbase; // color on-base flags
  reg rclamp, gclamp, bclamp; // color clamps, true when gain (including gainbase) is max
  wire pullback; // pullback when none on-base
  wire rpeg, gpeg, bpeg; // peg when only hot channel is on-base
  wire covflow; // transient on-base and overflow flags
  wire hotbit; // current bit of multiplier coef
  reg normal; // normal mode, max channel on base, no pegs or pullback
  reg g_grtrthn_b, r_ismax, r_ismin;  // used to store sort order
  reg stable; // closeness flag to suppress hunting
  wire [`TOPBIT+1:0] gnbasedash;
  wire [4:0] gnbasep1;
  wire [3:0] bitcount;
// DEV MOD 3 of 4: EXTRA DECLARATIONS
/*
  output [4:0] state;
  output [2:0] channel, cow;
  output [`TOPBIT+1:0] result;
  output stable, r_ismax, r_ismin, g_grtrthn_b;
  output ronbase, gonbase, bonbase, rclamp, gclamp, bclamp, rpeg, gpeg, bpeg, normal, pullback;
  reg subtract_ff;
  reg [`TOPBIT+1:0] op1_ff, op2_ff;
  output subtract_ff;
  output [`TOPBIT+1:0] op1_ff, op2_ff;
*/

// parse and scale the input gainbase (10 + input code, allowed to overflow)
// this lets us code a useful range of bases in 3 bits
/* 	assign gnbasep10 = 4'd10 + gnbase; */
/* 	assign gnbasep2 = 4'd2 + gnbase; */
	assign gnbasep1 = 4'd1 + gnbase;
// dash version of gainbase, sitting 1 bit from the top of a long 2's comp word
/* 	assign gnbasedash = {{1{gnbasep10[3]}},gnbasep10,`BOTBITP4'd0}; */
/* 	assign gnbasedash = {1'd0,gnbasep2,`BOTBITP4'd0}; */
/* 	assign gnbasedash = {gnbasep2,`BOTBITP4'd0,1'd0}; */
// new DVC-compatible gnbase, 0.5 < g <= 1.0, 4-bit code in
//	assign gnbasedash = {gnbasep1,`BOTBITP4'd0};
	assign gnbasedash = {gnbasep1,11'd0}; //Louis

// ALU function
  function [`TOPBIT+1:0] alu;
    input subtract;
    input [`TOPBIT+1:0] op1, op2;
    alu = subtract ? (op1 - op2) : (op1 + op2);
  endfunction

// speedup when no-op, also account for 1+c coding of on-diagonal matrix elements
/*   assign bitcount = remote ? 1 : (cow == 0) ? (`BITPLANES + 1) : `BITPLANES; */
  assign bitcount = remote ? 1 : `BITPLANES;

// controlling state machine 
  always @ (posedge clk or negedge resetb) begin
    if (!resetb) begin
      state <= 0;
      g_grtrthn_b <= 0;
      r_ismin <= 0;
      r_ismax <= 0;
      result = 0;
      rclamp <= 0;
      gclamp <= 0;
      bclamp <= 0;
      ronbase <= 1;
      gonbase <= 1;
      bonbase <= 1;
      rsum <= `ACBALLONES;
      gsum <= `ACBALLONES;
      bsum <= `ACBALLONES;
      m_rinr <= 192;
      m_ginr <= 0;
      m_binr <= 0;
      m_ring <= 0;
      m_ging <= 192;
      m_bing <= 0;
      m_rinb <= 0;
      m_ginb <= 0;
      m_binb <= 192;
      gain_plus_gnbase <= 0;
      cow = 0;
      channel = 0;
      bitplane = 0;
      stable = 0;
    end else begin
      
      // definition of the controlling state machine
      case (state)

        1: begin
             // [gacc-bacc] sort calculation
             op1 = bacc; // U
             op2 = gacc; // U
             subtract = 1;
             result = alu(subtract,op1,op2); // S
             g_grtrthn_b <= result[`TOPBIT+1]; // store result of sort 1
             state <= 11;
           end  // of state 1

        11: begin
             // suppress small g-b differences for no-hunt
             op1 = (racc >> 6); // U
             op2 = result; // S
             subtract = !g_grtrthn_b;
             result = alu(subtract,op1,op2); // S
      		 stable = !result[`TOPBIT+1];
             state <= 2;
           end  // of state 11

        2: begin
             // [racc-min(gacc,bacc)] sort calculation
             op1 = racc; // U
             op2 = g_grtrthn_b ? bacc : gacc; // U
             subtract = 1;
             result = alu(subtract,op1,op2); // S
             r_ismin <= result[`TOPBIT+1]; // store result of sort 2
             state <= stable ? 12 : 3;
           end  // of state 2

        12: begin
             op1 = (racc >> 6); // U
             op2 = result; // S
             subtract = !r_ismin;
             result = alu(subtract,op1,op2); // S
             // conditional on stability criterion, suppress updates
      		 stable = !result[`TOPBIT+1];
             state <= stable ? 6 : 3;
           end  // of state 12

        3: begin
             // [racc-max(gacc,bacc)] sort calculation
             op1 = racc; // U
             op2 = g_grtrthn_b ? gacc : bacc; // U
             subtract = 1;
             result = alu(subtract,op1,op2); // S
             r_ismax <= !result[`TOPBIT+1]; // store result of sort 3
             state <= 4;
           end  // of state 3

        4: begin
             // channel 'step' calculation
             if(normal) begin // max channel on base, others floating
               case ({r_ismax,r_ismin,g_grtrthn_b})
                 0: op1 = bacc;  // sort gave gacc as min, bacc as max
                 1: op1 = gacc;  // sort gave bacc as min, gacc as max
                 2: op1 = bacc;  // sort gave racc as min, bacc as max
                 3: op1 = gacc;  // sort gave racc as min, gacc as max
                 4: op1 = racc;  // sort gave gacc as min, racc as max
                 default: op1 = racc;  // sort gave bacc as min, racc as max
               endcase
             end else begin // pegging or pulling back
               case (channel)
                 0:  case ({r_ismax,r_ismin,g_grtrthn_b})
                       0: op1 = (!pullback && rpeg) ? racc : (gpeg ? gacc : bacc);
                       1: op1 = (!pullback && rpeg) ? racc : (gpeg ? gacc : bacc);
                       2: op1 = (!pullback && rpeg) ? racc : bacc;
                       3: op1 = (!pullback && rpeg) ? racc : gacc;
                       4: op1 = (!pullback && rpeg) ? racc : gacc;
                       default: op1 = (!pullback && rpeg) ? racc : bacc;
                     endcase
                 1:  case ({r_ismax,r_ismin,g_grtrthn_b})
                       0: op1 = (!pullback && gpeg) ? gacc : bacc;
                       1: op1 = (!pullback && gpeg) ? gacc : bacc;
                       2: op1 = (!pullback && gpeg) ? gacc : (rpeg ? racc : bacc);
                       3: op1 = (!pullback && gpeg) ? gacc : racc;
                       4: op1 = (!pullback && gpeg) ? gacc : racc;
                       default: op1 = (!pullback && gpeg) ? gacc : (rpeg ? racc : bacc);
                     endcase
                 default:  case ({r_ismax,r_ismin,g_grtrthn_b})
                       0: op1 = (!pullback && bpeg) ? bacc : gacc;
                       1: op1 = (!pullback && bpeg) ? bacc : gacc;
                       2: op1 = (!pullback && bpeg) ? bacc : racc;
                       3: op1 = (!pullback && bpeg) ? bacc : (rpeg ? racc : gacc);
                       4: op1 = (!pullback && bpeg) ? bacc : (rpeg ? racc : gacc);
                       default: op1 = (!pullback && bpeg) ? bacc : racc;
                     endcase
               endcase // U
             end
             case (channel) // op2 is simply the hot channel
               0: op2 = racc;
               1: op2 = gacc;
               default: op2 = bacc;
             endcase // U
             subtract = 1;
             result = alu(subtract,op1,op2); // S
             state <= 5;
           end  // of state 4
         
        5: begin
             // addition of step to channel energy-difference accumulator
             case (channel)
               0: op1 = rsum;
               1: op1 = gsum;
               default: op1 = bsum;
             endcase // S
             case (mu)
/*
               0: op2 = {result[`TOPBIT-3:0],4'd0};
               1: op2 = {result[`TOPBIT-2:0],3'd0};
               2: op2 = {result[`TOPBIT-1:0],2'd0};
               3: op2 = {result[`TOPBIT:0],1'd0};
               4: op2 = result;
               5: op2 = {{2{result[`TOPBIT+1]}},result[`TOPBIT:1]};
               6: op2 = {{3{result[`TOPBIT+1]}},result[`TOPBIT:2]};
               7: op2 = {{4{result[`TOPBIT+1]}},result[`TOPBIT:3]};
*/
               0: op2 = result;
               1: op2 = {{2{result[`TOPBIT+1]}},result[`TOPBIT:1]};
               2: op2 = {{3{result[`TOPBIT+1]}},result[`TOPBIT:2]};
               3: op2 = {{4{result[`TOPBIT+1]}},result[`TOPBIT:3]};
               4: op2 = {{5{result[`TOPBIT+1]}},result[`TOPBIT:4]};
               5: op2 = {{6{result[`TOPBIT+1]}},result[`TOPBIT:5]};
               6: op2 = {{7{result[`TOPBIT+1]}},result[`TOPBIT:6]};
               7: op2 = {{8{result[`TOPBIT+1]}},result[`TOPBIT:7]};
               default: op2 = {result[`TOPBIT-7:0],8'd0};
/*
               0: op2 = result;
               1: op2 = result;
               2: op2 = result;
               3: op2 = result;
               4: op2 = result;
               5: op2 = result;
               6: op2 = result;
               7: op2 = result;
*/
             endcase // S
             subtract = 0;
             result = alu(subtract,op1,op2); // S
             case (channel) // to store new accumulator value
               0: rsum <= (!op2[`TOPBIT+1] & (rclamp | covflow)) ? rsum : result;
               1: gsum <= (!op2[`TOPBIT+1] & (gclamp | covflow)) ? gsum : result;
               default: bsum <= (!op2[`TOPBIT+1] & (bclamp | covflow)) ? bsum : result;
             endcase
             case (channel) // on-base flag for peg/pullback decisions
               0: ronbase <= result[`TOPBIT+1];
               1: gonbase <= result[`TOPBIT+1];
               default: bonbase <= result[`TOPBIT+1];
             endcase
             state <= 6;
           end  // of state 5
 
        6: begin
             // add gainbase to channel gain
             case (channel)
               0: op1 = rsum;
               1: op1 = gsum;
               default: op1 = bsum;
             endcase // S
             subtract = 0;
             op2 = gnbasedash; // U
             subtract = 0;
             result = alu(subtract,op1,op2); // U
/*              gain_plus_gnbase <= result[`TOPBIT+1] ? `MAXPOS : result; */
             gain_plus_gnbase <= result;
             state <= 8; 
           end // of state 6

        8: begin
             // bitplane of shift-add multiply computation
             op1 = bitplane ? {1'd0,result[`TOPBIT+1:1]} : `ROUNDINGBIT; // S
/*
             op2 = remote ? {1'd0,ielement,`BOTBIT'd0}
                          : ((((bitplane + 1) == (`BITPLANES + 1)) || ielement[bitplane])?
                                                                            gain_plus_gnbase : 0); // S
*/
//             op2 = remote ? {1'd0,ielement,`BOTBIT'd0} : (hotbit ? gain_plus_gnbase : 0); // S
	     op2 = remote ? {1'd0,ielement,7'd0} : (hotbit ? gain_plus_gnbase : 0); // S //Louis
             subtract = ((cow != 0) && ((bitplane + 1) == `BITPLANES));
             result = alu(subtract,op1,op2); // S
	         if ((bitplane + 1) == bitcount) begin
               // dump results
               case(cow)
                 0: outword = remote ? result[`TOPBIT:`BOTBIT]
                                            : (result[`TOPBIT+1] ? `MAXPOS
                                                  : {!result[`TOPBIT],result[`TOPBIT-1:`BOTBIT]});
           default: outword = remote ? result[`TOPBIT:`BOTBIT]
                                            : (result[`TOPBIT+1] ?
                                                  (result[`TOPBIT] ? result[`TOPBIT:`BOTBIT] : `MAXNEG)
                                                : (result[`TOPBIT] ? `MAXPOS : result[`TOPBIT:`BOTBIT]));
               endcase
               // clip & clamp from on-diagonal outputs, only clip off-diagonal outputs
               case (channel)   
                 0: case (cow)
                      0: begin
                           m_rinr <= outword;
                           rclamp <= result[`TOPBIT+1];
                         end
                      1: begin
                           if(pregain)
                             m_ring <= outword;
                           else
                             m_ginr <= outword;
                         end
                      default: begin
                           if(pregain)
                             m_rinb <= outword;
                           else
                             m_binr <= outword;
                         end
                    endcase
                 1: case (cow)
                      0: begin
                           m_ging <= outword;
                           gclamp <= result[`TOPBIT+1];
                         end
                      1: begin
                           if(pregain)
                             m_ginb <= outword;
                           else
                             m_bing <= outword;
                         end
                      default: begin
                           if(pregain)
                             m_ginr <= outword;
                           else
                             m_ring <= outword;
                         end
                    endcase
                 default: case (cow)
                      0: begin
                           m_binb <= outword;
                           bclamp <= result[`TOPBIT+1];
                         end
                      1: begin
                           if(pregain)
                             m_binr <= outword;
                           else
                             m_rinb <= outword;
                         end
                      default: begin
                           if(pregain)
                             m_bing <= outword;
                           else
                             m_ginb <= outword;
                         end
                    endcase
               endcase
		       bitplane = 0;
               // manage channel & cow transitions
               if (((cow + 1) == 3) || (((cow + 1) == 1) && result[`TOPBIT+1])) begin
		         cow = 0;
                 if ((channel + 1) == 3) begin
                   state <= 0;
                 end else begin
                   channel = channel + 1;
                   state <= remote ? 8 : (stable ? 6 : 4);
                 end
               end else
		         cow = cow + 1;
             end else
		       bitplane = bitplane + 1;
           end // of state 8
           
        default: begin
             /* default idle state (includes state 0) */
             // reset the entire machine
             channel = 0;
             cow = 0;
             bitplane = 0;
             // setup a null operation for the adder/subtracter
             // (this calculation will be performed throughout the
             //  entire active video field)
             op1 = 0;
             op2 = 0;
             subtract = 0;
             result = alu(subtract,op1,op2);
             state <= (run & !freeze) ? (remote ? 8 : 1) : 0; 
             rsum <= remote ? `ACBALLONES : rsum;
             gsum <= remote ? `ACBALLONES : gsum;
             bsum <= remote ? `ACBALLONES : bsum;
           end
        
        endcase // of state machine
      
// DEV MOD 4 of 4: EXTRA REGISTER ASSIGNMENTS
/*
		op1_ff <= op1;
		op2_ff <= op2;
		subtract_ff <= subtract;
*/

    end // of non-reset behaviour 

  end // of state machine

// mult bit
  assign hotbit = (((bitplane + 1) == `BITPLANES) && (cow == 0)) ?
                       !ielement[bitplane] : ielement[bitplane];

// encoded controls off on-base flags
  assign pullback = !(ronbase | gonbase | bonbase);
  assign rpeg = !(gonbase | bonbase);
  assign gpeg = !(ronbase | bonbase);
  assign bpeg = !(ronbase | gonbase);
  always @ (r_ismax or r_ismin or g_grtrthn_b or ronbase or gonbase or bonbase) begin
    case ({r_ismax,r_ismin,g_grtrthn_b})
      0: normal = bonbase;  // sort gave gacc as min, bacc as max
      1: normal = gonbase;  // sort gave bacc as min, gacc as max
      2: normal = bonbase;  // sort gave racc as min, bacc as max
      3: normal = gonbase;  // sort gave racc as min, gacc as max
      4: normal = ronbase;  // sort gave gacc as min, racc as max
      default: normal = ronbase;  // sort gave bacc as min, racc as max
    endcase
  end

// encoded arithmetic status
/*   assign conbase = (result[`TOPBIT+1] & result[`TOPBIT] & result[`TOPBIT-1]); */
  assign covflow = (!result[`TOPBIT+1] & result[`TOPBIT] & !result[`TOPBIT-1]);

// mux to route the correct element from the input matrix into the multiplier
  always @ (channel or cow or rinr or ginr or binr or ring or ging or bing or
			 rinb or ginb or binb or pregain) begin
    case (channel)   
      0: case (cow)
           0: ielement = rinr;
           1: ielement = (!pregain? ginr : ring);
           default: ielement = (!pregain? binr : rinb);
         endcase
      1: case (cow)
            0: ielement = ging;
            1: ielement = (!pregain? bing : ginb);
            default: ielement = (!pregain? ring : ginr);
         endcase
      default: case (cow)
            0: ielement = binb;
            1: ielement = (!pregain? rinb : binr);
            default: ielement = (!pregain? ginb : bing);
         endcase
    endcase // of input matrix element mux
  end

endmodule // of colctrl
