/*
 *  Description:  Camera Embedded seq. decoder
 *
 *  Designer : Paul Likoudis
 *  Company  : VLSI Vision Ltd.
 *  Module   : BYTEDEC
 *
 * Revision History:
 * DD-MMM-YY:  Comment
 * 21-MAY-97:  Copy from CPiA original
*/

module BYTEDEC (
                vp_clk, 
                enph0, 
                enph2, 
                rstxxa_n, 
                rx_sftrst_n, 
                NO_CAMERA, 
                DATA, 
                NIBSYNC, 
                NEWSYNC, 
                FRAMEERR,
                BYTE_ERROR, 
                LINE_NUMBER, 
                LINE_CODE,	
                DATA_OUT,
                DATA_VALID, 
                DATA_IMMINENT, 
                SOBL, 
                SOBK, 
                SOVL, 
                SOF, 
                EOF, 
                SOFDATA_OUT, 
                SOFDATA_VALID,
                SOFDATA_IMMINENT, 
                COUNT, 
                FASTFORWARDMODE
               );
input  [7:0] DATA;
input        vp_clk, 
             enph0, 
             enph2, 
             rstxxa_n, 
             rx_sftrst_n, 
             NO_CAMERA, 
             NIBSYNC, 
             NEWSYNC, 
             FRAMEERR, 
             FASTFORWARDMODE;
output [7:0] DATA_OUT, 
             SOFDATA_OUT;
output [2:0] LINE_CODE;
output [8:0] LINE_NUMBER, 
             COUNT;
output       DATA_VALID, 
             DATA_IMMINENT;
output       SOBL, 
             SOBK, 
             SOVL, 
             SOF, 
             EOF;
output       BYTE_ERROR, 
             SOFDATA_VALID, 
             SOFDATA_IMMINENT;

reg    [7:0] DATA_OUT, 
             SOFDATA_OUT;
reg    [2:0] LINE_CODE;
reg    [2:0] DELCMD_CODE;
reg    [2:0] LASTVAL;
reg    [8:0] LINENUM,
             LINE_NUMBER,
             NEXTLINE,
             COUNT;
reg          EOL, 
             SOBL, 
             SOBK, 
             SOVL, 
             SOF, 
             EOF;
reg          OUTPUTDATA, 
             DELSUP2, 
             SUP2DEL, 
             PIXVALID;
reg          CMDERROR, 
             BYTE_ERROR, 
             SMERROR, 
             LINEERROR;
reg          SOFOUTPUTDATA, 
             SOFPIXVALID, 
             SOFIMM, 
             DELVALID;
reg          DISCARDLINEERROR;

wire         ERROR, 
             SOFRESET_n;
wire   [3:0] CMDOUT;
//wire [3:0] CMD_CODE;
wire         VALID, 
             NIBSYNC, 
             NEWSYNC;
wire         LINEPAR, 
             SMRESET_n, 
             DATA_ERROR;
wire         COMMAND, 
             EOLCMD, 
             SUP1, 
             EOLSUP1, 
             SUP2, 
             EOLSUP2;

wire         BYTERESET_n = rx_sftrst_n & (~NO_CAMERA);

//assign SMRESETB = RESETB & (~CMDERROR & ~SMERROR);
assign SMRESET_n = BYTERESET_n & (~CMDERROR & ~SMERROR);
assign ERROR = CMDERROR | LINEERROR | DATA_ERROR | FRAMEERR;

BYTESM   sm (
             .NEWSYNC(NEWSYNC), 
             .vp_clk(vp_clk), 
             .enph0(enph0), 
             .rstxxa_n(rstxxa_n), 
             .SMRESET_n(SMRESET_n), 
             .BYTE_ERROR(DATA_ERROR), 
             .DATA_VALID(VALID), 
             .COMMAND(COMMAND), 
             .SUP1(SUP1), 
             .SUP2(SUP2), 
             .EOLCMD(EOLCMD), 
             .EOLSUP1(EOLSUP1), 
             .EOLSUP2(EOLSUP2)
            );

CMDPAR  cmd (
             .DATAIN(DATA), 
             .DATAOUT(CMDOUT), 
             .ERROR2BIT(ERROR2BIT)
            );

PARITY6 par (
             .DIN(DATA[6:1]), 
             .P(DATA[0]), 
             .O(LINEPAR)
            );

// Error handling - latch error signals to be cleared by start of frame
// If sm byte-error flagged then reset sm, then check for command - if EOL
// and not EOLCMD then reset again - this will not matter as we will have
// junked frame anyway. 

always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n) 
         SMERROR <= 1'b0;
      else
         if (!BYTERESET_n) 
            SMERROR <= 1'b0;
         else
            if (enph0)
               SMERROR <= DATA_ERROR;
            else
               SMERROR <= SMERROR;
   end

// Block out byte error as I don't want to stop DATA_ERROR or CMDERROR so 
// that state machine is reset.
always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n) 
         BYTE_ERROR <= 1'b0;
      else
         if (!BYTERESET_n) 
            BYTE_ERROR <= 1'b0;
         else
            if (enph0)
               if (SOF || FASTFORWARDMODE)
                  BYTE_ERROR <= 1'b0;
               else
                  if (ERROR)
                     BYTE_ERROR <= 1'b1;
                  else
                     BYTE_ERROR <= BYTE_ERROR;
            else
                     BYTE_ERROR <= BYTE_ERROR;
   end

always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         begin
            EOL      <= 1'b0;
            SOBL     <= 1'b0;
            SOBK     <= 1'b0;
            SOVL     <= 1'b0;
            SOF      <= 1'b0;
            EOF      <= 1'b0;
            CMDERROR <= 1'b0;
         end
      else
         if (!BYTERESET_n)
            begin
               EOL      <= 1'b0;
               SOBL     <= 1'b0;
               SOBK     <= 1'b0;
               SOVL     <= 1'b0;
               SOF      <= 1'b0;
               EOF      <= 1'b0;
               CMDERROR <= 1'b0;
            end
         else
            if (enph0)
               begin
                  if (COMMAND && !ERROR2BIT)
                     begin
                        case (CMDOUT)
                           4'b1000:
                              begin
                                 // EndOfLine 
                                 EOL  <= 1'b0;
                                 SOBL <= 1'b0;
                                 SOBK <= 1'b0;
                                 SOVL <= 1'b0;
                                 SOF  <= 1'b0;
                                 EOF  <= 1'b0;
                                 CMDERROR <= 1'b1;
                              end
                           4'b1001:
                              begin
                                 // Blank Line 
                                 EOL  <= 1'b0;
                                 SOBL <= 1'b1;
                                 SOBK <= 1'b0;
                                 SOVL <= 1'b0;
                                 SOF  <= 1'b0;
                                 EOF  <= 1'b0;
                                 CMDERROR <= 1'b0;
                              end
                           4'b1010:
                              begin
                                 // Black Line 
                                 EOL  <= 1'b0;
                                 SOBL <= 1'b0;
                                 SOBK <= 1'b1;
                                 SOVL <= 1'b0;
                                 SOF  <= 1'b0;
                                 EOF  <= 1'b0;
                                 CMDERROR <= 1'b0;
                              end
                           4'b1011:
                              begin
                                 // Visible Line 
                                 EOL  <= 1'b0;
                                 SOBL <= 1'b0;
                                 SOBK <= 1'b0;
                                 SOVL <= 1'b1;
                                 SOF  <= 1'b0;
                                 EOF  <= 1'b0;
                                 CMDERROR <= 1'b0;
                              end
                           4'b1100:
                              begin
                                 // Start of Frame 
                                 EOL  <= 1'b0;
                                 SOBL <= 1'b0;
                                 SOBK <= 1'b0;
                                 SOVL <= 1'b0;
                                 SOF  <= 1'b1;
                                 EOF  <= 1'b0;
                                 CMDERROR <= 1'b0;
                              end
                           4'b1101:
                              begin
                                 // End of Frame 
                                 EOL  <= 1'b0;
                                 SOBL <= 1'b0;
                                 SOBK <= 1'b0;
                                 SOVL <= 1'b0;
                                 SOF  <= 1'b0;
                                 EOF  <= 1'b1;
                                 CMDERROR <= 1'b0;
                              end
                           4'b1110 , 
                           4'b1111 :
                              begin
                                 // Reserved - Unused 
                                 EOL  <= 1'b0;
                                 SOBL <= 1'b0;
                                 SOBK <= 1'b0;
                                 SOVL <= 1'b0;
                                 SOF  <= 1'b0;
                                 EOF  <= 1'b0;
                                 CMDERROR <= 1'b0;
                              end
                           default :
                              begin
                                 // Bit 7 not set - error
                                 EOL  <= 1'b0;
                                 SOBL <= 1'b0;
                                 SOBK <= 1'b0;
                                 SOVL <= 1'b0;
                                 SOF  <= 1'b0;
                                 EOF  <= 1'b0;
                                 CMDERROR <= 1'b1;
                              end
                        endcase
                     end
                  else
                     begin
                        if (EOLCMD && !ERROR2BIT)
                           begin
                              case (CMDOUT)
                                 4'b1000:
                                    begin
                                    // EndOfLine - only valid cmd 
   		       	            EOL  <= 1'b1;
   		       	            SOBL <= 1'b0;
   		       	            SOBK <= 1'b0;
   		       	            SOVL <= 1'b0;
   		       	            SOF  <= 1'b0;
   		       	            EOF  <= 1'b0;
   		       	            CMDERROR <= 1'b0;
                                    end
                                 default :
                                    begin
   		       	            EOL  <= 1'b0;
   		       	            SOBL <= 1'b0;
   		       	            SOBK <= 1'b0;
   		       	            SOVL <= 1'b0;
   		       	            SOF  <= 1'b0;
   		       	            EOF  <= 1'b0;
   		       	            CMDERROR <= 1'b1;
                                    end
                              endcase
                           end
                        else
                           begin
                              if ((COMMAND || EOLCMD) && ERROR2BIT)
                                 begin
                                    CMDERROR <= 1'b1;
                                    EOL  <= 1'b0;
                                    SOBL <= 1'b0;
                                    SOBK <= 1'b0;
                                    SOVL <= 1'b0;
                                    SOF  <= 1'b0;
                                    EOF  <= 1'b0;
                                 end
                              else
                                 begin
                                    CMDERROR <= 1'b0;
                                    EOL  <= 1'b0;
                                    SOBL <= 1'b0;
                                    SOBK <= 1'b0;
                                    SOVL <= 1'b0;
                                    SOF  <= 1'b0;
                                    EOF  <= 1'b0;
                                 end
                           end
                     end
               end
            else
               begin
                  EOL      <= EOL;
                  SOBL     <= SOBL;
                  SOBK     <= SOBK;
                  SOVL     <= SOVL;
                  SOF      <= SOF;
                  EOF      <= EOF;
                  CMDERROR <= CMDERROR;
               end
   end
   
//assign CMD_CODE = CMDOUT[7:4];
   
// Added SOFOUTPUTDATA to signal when SOF line is here
always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         begin
            OUTPUTDATA    <= 1'b0;
            SOFOUTPUTDATA <= 1'b0;
         end
      else
         if (!BYTERESET_n)
            begin
               OUTPUTDATA    <= 1'b0;
               SOFOUTPUTDATA <= 1'b0;
            end
         else
            if (enph0)
               if (NEWSYNC)
                  begin 
                     OUTPUTDATA    <=  1'b0;
                     SOFOUTPUTDATA <=  1'b0;
                  end
               else
                  if (COMMAND)
                     begin
                        OUTPUTDATA    <= ~CMDOUT[2] & CMDOUT[1];
                        SOFOUTPUTDATA <= CMDOUT[2] & ~CMDOUT[1] & ~CMDOUT[0];
                     end
                  else
                     begin
                        OUTPUTDATA    <= OUTPUTDATA;
                        SOFOUTPUTDATA <= SOFOUTPUTDATA;
                     end
            else
               begin
                  OUTPUTDATA    <= OUTPUTDATA;
                  SOFOUTPUTDATA <= SOFOUTPUTDATA;
               end
   end
   
always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         begin
            DELSUP2 <= 1'b0;
            SUP2DEL <= 1'b0;
         end
      else
         if (!BYTERESET_n)
            begin
               DELSUP2 <= 1'b0;
               SUP2DEL <= 1'b0;
            end
         else
            if (enph0)
               begin
                  DELSUP2 <= SUP2DEL;
                  SUP2DEL <= SUP2;
               end
            else
               begin
                  DELSUP2 <= DELSUP2;
                  SUP2DEL <= SUP2DEL;
               end
   end

always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         begin
            DATA_OUT <= 8'b00000000;
            PIXVALID <= 1'b0;
         end
      else
         if (!BYTERESET_n)
            begin
               DATA_OUT <= 8'b00000000;
               PIXVALID <= 1'b0;
            end
         else
            if (enph0)
               if (NIBSYNC)
                  begin
                     if (COMMAND)
                        begin
                           DATA_OUT <= 8'b00000000;
                           PIXVALID <= 1'b0;
                        end
                     else
                        if (OUTPUTDATA && VALID)
                           begin
                              if (DATA == 8'b11111111 | DATA == 8'b00000000)
                                 begin
                                    DATA_OUT <= 8'b00000000;
                                    PIXVALID <= 1'b0;
                                 end
                              else
                                 begin
                                    DATA_OUT <= DATA;
                                    PIXVALID <= 1'b1;
                                 end
                           end
                        else
                           begin
                              DATA_OUT <= 8'b00000000;
                              PIXVALID <= 1'b0;
                           end
                  end
               else
                  begin
                     DATA_OUT <= DATA_OUT;
                     PIXVALID <= 1'b0;
                  end
            else
               begin
                  DATA_OUT <= DATA_OUT;
                  PIXVALID <= PIXVALID;
               end
   end

always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         begin
            LINE_CODE   <= 3'b111;
            DELCMD_CODE <= 3'b111;
         end
      else
         if (!BYTERESET_n)
            begin
               LINE_CODE   <= 3'b111;
               DELCMD_CODE <= 3'b111;
            end
         else
            if (enph0)
               if (NIBSYNC)
                  if (COMMAND)
                     begin
                        LINE_CODE   <= LINE_CODE; 
                        DELCMD_CODE <= CMDOUT[2:0];
                     end
                  else
                     begin
                        LINE_CODE   <= DELCMD_CODE;
                        DELCMD_CODE <= DELCMD_CODE;
                     end
               else
                  begin
                     LINE_CODE   <= LINE_CODE;
                     DELCMD_CODE <= DELCMD_CODE;
                  end
            else
               begin
                  LINE_CODE   <= LINE_CODE;
                  DELCMD_CODE <= DELCMD_CODE;
               end
   end

always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         LINE_NUMBER <= 9'b000000000;
      else
         if (!BYTERESET_n)
            LINE_NUMBER <= 9'b000000000;
         else
            if (enph0)
               if (NIBSYNC)
                  if (DELSUP2)
                     LINE_NUMBER <= LINENUM;
                  else
                     LINE_NUMBER <= LINE_NUMBER;
               else
                  LINE_NUMBER <= LINE_NUMBER;
           else
               LINE_NUMBER <= LINE_NUMBER;
   end

assign DATA_VALID = PIXVALID & enph2;
assign DATA_IMMINENT = OUTPUTDATA & SUP2DEL & enph2;
// Valid and imminent signals for SOF data 
assign SOFDATA_VALID = SOFPIXVALID & enph2;
assign SOFDATA_IMMINENT = SOFIMM & enph2;
// Line data should be reset every frame
//assign SOFRESETB = RESETB & ~(SOF & ~NIBSYNC);
assign SOFRESET_n = BYTERESET_n & ~(SOF & ~NIBSYNC);
  
always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         NEXTLINE <= 9'b000000000;
      else
         if (!SOFRESET_n)
            NEXTLINE <= 9'b000000000;
         else
            if (enph0)
               if (EOL)
                  NEXTLINE <= NEXTLINE + 1;
               else
                  NEXTLINE <= NEXTLINE;
            else
               NEXTLINE <= NEXTLINE;
   end

// LineError should be parity problem or non end of line correct or
// non consecutive line
always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         begin
            LINEERROR <= 1'b0;
            LASTVAL   <= 3'b000;
            LINENUM   <= 9'b000000000;
         end
      else
         if (!SOFRESET_n)
            begin
               LINEERROR <= 1'b0;
               LASTVAL   <= 3'b000;
               LINENUM   <= 9'b000000000;
            end
         else
            if (enph0)
               if (NIBSYNC)
                  begin
                     if (SUP1)
                        begin
                           if (LINEPAR || FASTFORWARDMODE)
                              begin
                                 LINEERROR <= 1'b0;
                                 LASTVAL   <= DATA[3:1];
                                 LINENUM   <= LINENUM;
                              end
                           else
                              begin
                                 LINEERROR <= 1'b1;
                                 LASTVAL   <= LASTVAL;
                                 LINENUM   <= LINENUM;
                              end
                        end
                     else
                        if (SUP2)
                           begin
                              if (LINEPAR || FASTFORWARDMODE)
                                 begin
                                    LINEERROR <= 1'b0;
                                    LASTVAL   <= LASTVAL;
                                    LINENUM   <= {LASTVAL,DATA[6:1]};
                                 end
                              else
                                 begin
                                    LINEERROR <= 1'b1;
                                    LASTVAL   <= LASTVAL;
                                    LINENUM   <= LINENUM;
                                 end
                           end
                        else
                           if (DELSUP2)
                              begin
                                 if (LINENUM == NEXTLINE)
                                    begin
                                       LINEERROR <= 1'b0;
                                       LASTVAL   <= LASTVAL;
                                       LINENUM   <= LINENUM;
                                    end
                                 else
                                    if (FASTFORWARDMODE)
                                       begin
                                          LINEERROR <= 1'b0;
                                          LASTVAL   <= LASTVAL;
                                          LINENUM   <= LINENUM;
                                       end
                                    else
                                       begin
                                          LINEERROR <= 1'b1;
                                          LASTVAL   <= LASTVAL;
                                          LINENUM   <= LINENUM;
                                       end
                              end
                           else
                              if (EOLSUP1 | EOLSUP2)
                                 begin
                                    if (&DATA || FASTFORWARDMODE)
                                       begin
                                          LINEERROR <= 1'b0;
                                          LASTVAL   <= LASTVAL;
                                          LINENUM   <= LINENUM;
                                       end
                                    else
                                       begin
                                          LINEERROR <= 1'b1;
                                          LASTVAL   <= LASTVAL;
                                          LINENUM   <= LINENUM;
                                       end
                                 end
                              else
                                 begin
                                    LINEERROR <= 1'b0;
                                    LASTVAL   <= LASTVAL;
                                    LINENUM   <= LINENUM;
                                 end
                  end
               else
                  begin
                     LINEERROR <= 1'b0;
                     LASTVAL   <= LASTVAL;
                     LINENUM   <= LINENUM;
                  end
            else
               begin
                  LINEERROR <= LINEERROR;
                  LASTVAL   <= LASTVAL;
                  LINENUM   <= LINENUM;
               end
   end

// Create imminent signal on count value before data output, ie count = 0
// therefore need delvalid to flag when data is imminent
always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         SOFIMM <= 1'b0;
      else
         if (!BYTERESET_n)
            SOFIMM <= 1'b0;
         else
            if (enph0)
               if (NIBSYNC && SOFOUTPUTDATA)
                  if (COUNT == 9'd0 && DELVALID)
                     SOFIMM <= 1'b1;
                  else
                     SOFIMM <= 1'b0;
               else
                  SOFIMM <= 1'b0;
            else
               SOFIMM <= SOFIMM;
   end

always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         begin
            SOFDATA_OUT <= 8'b00000000;
            SOFPIXVALID <= 1'b0;
         end
      else
         if (!BYTERESET_n)
            begin
               SOFDATA_OUT <= 8'b00000000;
               SOFPIXVALID <= 1'b0;
            end
         else
            if (enph0)
               if (NIBSYNC)
                  begin
                     if (COMMAND)
                        begin
                           SOFDATA_OUT <= 8'b00000000;
                           SOFPIXVALID <= 1'b0;
                        end
                     else
                        if (((COUNT >= 9'd1) && (COUNT < 9'd258)) && SOFOUTPUTDATA)
                           begin
                              if (COUNT[0] == 1'b1)
                                 begin
                                    SOFDATA_OUT <= DATA;
                                    SOFPIXVALID <= 1'b1;
                                 end
                              else
                                 begin
                                    SOFDATA_OUT <= 8'b00000000;
                                    SOFPIXVALID <= 1'b0;
                                 end
                           end
                        else
                           begin
                              SOFDATA_OUT <= 8'b00000000;
                              SOFPIXVALID <= 1'b0;
                           end
                  end
               else
                  begin
                     SOFDATA_OUT <= SOFDATA_OUT;
                     SOFPIXVALID <= 1'b0;
                  end
            else
               begin
                  SOFDATA_OUT <= SOFDATA_OUT;
                  SOFPIXVALID <= SOFPIXVALID;
               end
   end

always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n)
         DELVALID <= 1'b0;
      else
         if (!BYTERESET_n)
            DELVALID <= 1'b0;
         else
            if (enph0)
               if (NIBSYNC)
                  DELVALID <= VALID;
               else
                  DELVALID <= DELVALID;
            else
               DELVALID <= DELVALID;
   end

// Count value will	count in pixels in start of frame line
always @ (posedge vp_clk or negedge rstxxa_n)
   begin
      if (!rstxxa_n) 
         COUNT <= 9'd0;
      else
         if (!BYTERESET_n) 
            COUNT <= 9'd0;
         else
            if (enph0)
               if (NIBSYNC) 
                  if ((SOFOUTPUTDATA | OUTPUTDATA) && DELVALID)
                     if (COUNT == 9'd356)
                        COUNT <= COUNT;
                     else
                        COUNT <= COUNT + 1;
                  else
                     COUNT <= 9'd0;
               else
                  COUNT <= COUNT;  
            else
               COUNT <= COUNT;  
   end        

endmodule 
