%%% A Square-Root Sampling Approach to Fast Histogram-Based Search
clear all
close all

quan = 4; % bin quantization, 8 and 4 are more suitable
nbin = ceil(256/quan); % number of bins

%% read the test image
test_images = {'vangogh.jpg', 'katsushika.jpg', 'seurat.jpg'};
disp( '---------------------------------------------------------------------------------------') 
disp('Test images:')
disp('[1] vangogh.jpg (767x600)  [2] katsushika.jpg (800x1060)  [3] seurat.jpg (750x501)');
choice = [input('Please choose a test image: '), 1]; choice = choice(1);
if ~ismember(choice, [1:4]), choice = 1; end
I_original = imread(['image/' test_images{choice}]);
I2 = double(I_original);
h = fspecial('gaussian',5,3); 
I2 = imfilter(I2,h); 

%% compute the bin index of each pixel in the test image
hI2 = (nbin^2)*floor(I2(:,:,1)/quan) + nbin*floor(I2(:,:,2)/quan) + floor(I2(:,:,3)/quan) + 1;
hI2 = int32(hI2); %% hI2 is an image consisting of the bin indices for the test image

%% parameters
num_box = 3;
e_wd = 3;

%% for different templates that have different sizes
switch choice
    case 1
        T_names ={'vangogh_1.jpg','vangogh_2.jpg','vangogh_3.jpg','vangogh_4.jpg','vangogh_5.jpg'};
    case 2
        T_names ={'katsushika_1.jpg','katsushika_2.jpg','katsushika_3.jpg','katsushika_4.jpg','katsushika_5.jpg','katsushika_6.jpg'};
    case 3
        T_names ={'seurat_1.jpg','seurat_2.jpg','seurat_3.jpg','seurat_4.jpg','seurat_5.jpg'};
end
num_T = numel(T_names) ;

disp('---')
disp('[1] Exact evaluation of the expected likelihood kernel (ELK)')
disp('[2] Sparse evaluation of the expected likelihood kernel (SELK)')
disp('[3*] Square-root sampling for sparse Bhattacharyya kernel evaluation (SBTCY)')
disp('[4] Exact evaluation of ELK SSD using the updating scheme (SSD_U))')
disp(['[5] Exact evaluation of Bhattacharyya kernel using the updating scheme (BTCY_U)'])
method = [input('Please select a search algorithm: (default [3]) '), 3]; method = method(1);
if ~ismember(method, [1:7]), method = 3; end

I_res = I_original;

support = cell(1,num_T);

fig1 = figure; set(fig1, 'Position', [50 150 500 100]);
set(fig1, 'Name', 'Template');
fig2 = figure;

disp('---')

%% the loop for different templates
for i = 1 : num_T 
    %% read the template image
    I = imread(['image/' T_names{i}]);
    figure(fig1);
    subplot(1, num_T, i);
    imshow(I);
    drawnow
    I = double(I);
    h = fspecial('gaussian',5,3); 
    I = imfilter(I,h); 
    
    %% compute the bin index of each pixel in the template image
    hI = (nbin^2)*floor(I(:,:,1)/quan) + nbin*floor(I(:,:,2)/quan) + floor(I(:,:,3)/quan) + 1;
    hI = int32(hI); %% hI is an image consisting of the bin indices for the template 

    %% compute the histogram for the template image
    p = hist(double(hI(:)), [1:nbin^3]);
    ratio = p./sum(p);

    %ratio = sqrt(p); % rejection sampling thresholds   
    disp(['Searching ' T_names{i} ' in ' test_images{choice} ' ...'])

    switch method
        case 1
            % ELK
            set(fig2, 'Name','ELK');
            tic
            [support] = ELK(ratio,hI,hI2);
            toc
            [I_res] = drawBoundingBox(I_res, support, size(hI), num_box, e_wd, [255 0 0]);
        case 2
            % SELK
            set(fig2, 'Name','SELK');
            tic
            [support] = SELK(ratio,hI,hI2);
            toc
            [I_res] = drawBoundingBox(I_res, support, size(hI), num_box, e_wd, [255 0 255]);
        case 3
            % SBTCY
            set(fig2, 'Name','SBTCY');
            tic
            [support] = SBTCY(sqrt(ratio),hI,hI2, int32(10));
            toc
            [I_res] = drawBoundingBox(I_res, support, size(hI), num_box, e_wd, [255 255 255]);    
        case 4
            % SSD_U
            set(fig2, 'Name','SSD_U');
            tic
            [support] = SSD_U(ratio,hI,hI2);
            toc
            [I_res] = drawBoundingBox(I_res, support, size(hI), num_box, e_wd, [0 255 255]);    
        case 5
            % BTCY_U
            set(fig2, 'Name','BTCY_U');
            tic
            [support] = BTCY_U(sqrt(ratio),hI,hI2);
            toc
            [I_res] = drawBoundingBox(I_res, support, size(hI), num_box, e_wd, [128 255 128]);
    end
    
    figure(fig2); 
    imshow(I_res);    
    drawnow
end
