#include <iostream.h>
#include <stdio.h>
#include <assert.h>
#include <float.h>
#include <string.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <GL/glut.h>

#include "io.h"

void colorBar(float f, unsigned char* RGB)
{
  unsigned char& Red   = RGB[0];
  unsigned char& Green = RGB[1];
  unsigned char& Blue  = RGB[2];

  if (f < 0) {
    Red = Green = Blue = 0;
  }
  else if (f <= 0.25) {  // Blue to Light Blue
    f *= 4.0;  // scale f to [0,1]
    Blue  = 255;
    Green = (int) (255.0 * sqrt(f));
    Red   = 0;
  }
  else if (f <= 0.50) { // Light Blue to Green
    f = (f * 4.0) - 1.0;  // scale f to [0,1]
    Blue  = (int) (255.0 * sqrt(1.0 - f));
    Green = 255;
    Red   = 0;
  }
  else if (f <= 0.75) { // Green to Yellow
    f = (f * 4.0) - 2.0;  // scale f to [0,1]
    Blue  = 0;
    Green = 255;
    Red   = (int) (255.0 * sqrt(f));
  }
  else if (f <= 1.0) { // Yellow to Red
    f = (f * 4.0) - 3.0; // scale f to [0,1]
    Blue  = 0;
    Green = (int) (255.0 * sqrt(1.0 - f));
    Red   = 255;
  }
  else if (f <= 2.0) { // Cyan to show the overflow
    f = f - 1.0; // scale f to [0,1]
    Blue = (int) (255.0 * sqrt(f));
    Green = 0;
    Red = 255;
  }
  else {
    Red = Blue = 255;
    Green = 0;
  }
}

vec3 dominantAxis(vec3 v)
{
  if (fabs(v[0]) >= fabs(v[1])) {
    if (fabs(v[0]) >= fabs(v[2]))
      return (v[0] >= 0) ? vec3(1,0,0) : vec3(-1,0,0);
    else
      return (v[2] >= 0) ? vec3(0,0,1) : vec3(0,0,-1);
  }
  else {
    if (fabs(v[1]) >= fabs(v[2]))
      return (v[1] >= 0) ? vec3(0,1,0) : vec3(0,-1,0);
    else
      return (v[2] >= 0) ? vec3(0,0,1) : vec3(0,0,-1);
  }
}

//===================================
//  class Camera
//===================================
void Camera::fromArray(float* arr)
{
  // arr[0..2]  is the center of projection.
  // arr[3..5]  is vector a (step size in X direction).
  // arr[6..8]  is vector b (step size in Y direction).
  // arr[9..11] is vector c (COP to origin of image plane).
  COP = vec3(arr[0], arr[1], arr[2]);
  a   = vec3(arr[3], arr[4], arr[5]);
  b   = vec3(arr[6], arr[7], arr[8]);
  c   = vec3(arr[9], arr[10], arr[11]);
}

void Camera::toArray(float* arr) 
{
  int i;
  
  for (i=0; i<3; i++) arr[i] = COP[i];
  for (i=0; i<3; i++) arr[3+i] = a[i];
  for (i=0; i<3; i++) arr[6+i] = b[i];
  for (i=0; i<3; i++) arr[9+i] = c[i];
}

void Camera::toMat3(mat3& P) const
{
  P = mat3(a, b, c).transpose();
}

void Camera::fromLookAt(vec3 eye, vec3 lookat, vec3 up, float fovY,
			int width, int height)
{
  fovY *= M_PI / 180;
  vec3 diff = lookat - eye;
  vec3 right = diff ^ up;
  vec3 down = -up;
  float pixelSize = diff.length() * tan(fovY*0.5) * 2 / height;
  
  right.normalize();
  down.normalize();
  COP = eye;
  a = right * pixelSize;
  b = down  * pixelSize;
  c = diff - a * (width*0.5) - b * (height*0.5);
}

void Camera::toLookAt(vec3& eye, vec3& lookat, vec3& up, float& fovY,
		      int width, int height)
{
  eye = COP;
  lookat = COP + c + a * (width*0.5) + b * (height*0.5);
  up = -b;
  up.normalize();
  fovY = 2 * atan(b.length()*height*0.5 / (lookat-eye).length());
  fovY *= 180.0 / M_PI;
}

void Camera::dump()
{
  printf("\nCOP=%f %f %f\n", COP[0], COP[1], COP[2]);
  printf("a=%f %f %f, len=%f\n",   a[0], a[1], a[2], a.length());
  printf("b=%f %f %f, len=%f\n",   b[0], b[1], b[2], b.length());
  printf("c=%f %f %f, len=%f\n", c[0], c[1], c[2], c.length());
}

//
// Camera::interpolate() produces smooth transition to the new camera
// setup (cam2) when t changes from 0 to 1.
// 
Camera Camera::interpolate(Camera cam2, float t)
{
  float yaw1, yaw2, dYaw, pitch1, pitch2, dPitch;
  vec3 c1, c2, dCOP;
  vec3 axisZ(0,0,1), axisY(0,1,0);
  mat4 rotation;
  Camera newCam;

  // Use (x,y) to find the yaw of old and new cameras.
  yaw1 = atan2f(c[1], c[0]);
  yaw2 = atan2f(cam2.c[1], cam2.c[0]);

  // Rotate the c vectors of both camera to the X/Z plane,
  // then use the (x,z) to find the difference of pitch.
  rotation = rotation3Drad(axisZ, -yaw1);
  c1 = rotation * c;
  pitch1 = atan2f(c1[0], c1[2]);
  rotation = rotation3Drad(axisZ, -yaw2);
  c2 = rotation * cam2.c;
  pitch2 = atan2f(c2[0], c2[2]);

  dCOP = cam2.COP - COP;
  dYaw   = yaw2 - yaw1;
  dPitch = pitch2 - pitch1;

  newCam.COP = COP + dCOP * t;
  // (1) rotate vectors a, b, and c by -yaw1 around Z axis
  rotation = rotation3Drad(axisZ, -yaw1);
  newCam.a = rotation * a;
  newCam.b = rotation * b;
  newCam.c = rotation * c;
  // (2) rotate vectors a, b, and c by dPitch*t around Y axis
  rotation = rotation3Drad(axisY, dPitch*t);
  newCam.a = rotation * newCam.a;
  newCam.b = rotation * newCam.b;
  newCam.c = rotation * newCam.c;
  // (3) rotate vectors a, b, and c by yaw1 + dYaw*t around Z axis
  rotation = rotation3Drad(axisZ, yaw1 + dYaw*t);
  newCam.a = rotation * newCam.a;
  newCam.b = rotation * newCam.b;
  newCam.c = rotation * newCam.c;

  return newCam;
}

//===================================
//  class DepthImage
//===================================
DepthImage::DepthImage()
{
  color = connectivity = 0;
  disparity = normal = 0;
}

DepthImage::~DepthImage()
{
  closeFile();
}

void DepthImage::saveRawFile(char* fname)
{
	FILE *fp = fopen(fname, "wb");
	float tmpCam[12];

	assert(fp);
	fwrite(&width,  sizeof(int), 1, fp);	
	fwrite(&height, sizeof(int), 1, fp);
	camera.toArray(tmpCam);
	fwrite(tmpCam, sizeof(float), 12, fp);	
	fwrite(color, 3, width*height, fp);	
	fwrite(disparity, sizeof(float), width*height, fp);
	fclose(fp);
}

void DepthImage::loadRawFile(char* fname)
{
	FILE *fp = fopen(fname, "rb");
	float tmpCam[12];

    // free the memory allocated for the previous input file
    closeFile();

	assert(fp);
	fread(&width,  sizeof(int), 1, fp);	
	fread(&height, sizeof(int), 1, fp);
	fread(tmpCam, sizeof(float), 12, fp);	
	camera.fromArray(tmpCam);
	color = new unsigned char[3*width*height];
	disparity = new float[width*height];
	fread(color, 3, width*height, fp);	
	fread(disparity, sizeof(float), width*height, fp);
	fclose(fp);
}

void DepthImage::closeFile()
{
  if(color) {
    delete[] color;
    if (connectivity) delete[] connectivity;
    if (disparity) delete[] disparity;
    color = 0;
	connectivity = 0;
    disparity = 0;
  }
  if (normal) {
    delete[] normal;
    normal=0;
  }
}

unsigned char* DepthImage::getColor(int x, int y) const
{
  return &color[3*(x+y*width)];
}

float* DepthImage::getNormal(int x, int y) const
{
  return &normal[3*(x+y*width)];
}

float DepthImage::getDisparity(int x, int y) const
{
  return disparity[x+y*width];
}

void DepthImage::setDisparity(int x, int y, float disp)
{
  disparity[x+y*width] = disp;
}

void DepthImage::generateNormals()
{
  vec3 p1, p2, diff;
  vec3 pDx, pDy, N, viewDir;
  float t1, t2, t, *pNormal;
  int i, x, y;

  assert(color && disparity);
  assert(width > 1 && height > 1);

  // get rid of the old data
  if (normal) delete[] normal;
  normal = new float[3*width*height];

  //=======================================================
  // The following method for detecting discontinuity is
  // now obsolete.  Kept for the record only.
  //=======================================================
  // The threshold for detecting discontinuity is
  // a function of the image resolution.
  // Let t = 1/disparity, (i.e. the ratio of the surface 
  // distance to the plane distance) then two points are
  // considered to be in the same surface if:
  //       (t1-t2)/t < camera.a.length()/d
  // where d is the projection of vector c onto the viewing
  // direction.
  //
  // threshold = camera.a.length() / (c*e3);

  for (y=0; y<height; y++) {
    for (x=0; x<width; x++) {
      // Special case for the background
      if (EPSILON > getDisparity(x, y)) {
	N = -(camera.a * x + camera.b * y + camera.c);
	N.normalize();
	pNormal = getNormal(x, y);  // get the pointer
	for (i=0; i<3; i++) {
	  pNormal[i] = N[i];
	}
	continue;
      }

      p1 = location_R3(x, y);

      //------- The X direction ------
      if (0 == x) {
	p2 = location_R3(x+1, y);
	pDx = p2 - p1;
	pDx.normalize();
      }
      else if (width-1 == x) {
	p2 = location_R3(x-1, y);
	pDx = p1 - p2;
	pDx.normalize();
      }
      else {
	// check the pixel on the left
	p2 = location_R3(x-1, y);
	pDx = p1 - p2;
	pDx.normalize();
	// check the pixel on the right
	p2 = location_R3(x+1, y);
	diff = p2 - p1;
	diff.normalize();

	// Check whether discontinuity occurs.
	if (pDx * diff < 0.714) {  // > 45 degree
	  // pick the one with less difference in disparity
	  t  = 1.0 / getDisparity(x, y);
	  t1 = 1.0 / getDisparity(x-1, y);
	  t2 = 1.0 / getDisparity(x+1, y);
	  if (fabs(t-t1) < fabs(t-t2)) {
	    // pDx is already computed.
	  }
	  else {
	    pDx = diff;
	  }
	}
	else { 
	  // Combine both derivatives.
	  pDx += diff;
	  pDx.normalize();
	}
      }

      //------- The Y direction ------
      if (0 == y) {
	p2 = location_R3(x, y+1);
	pDy = p2 - p1;
	pDy.normalize();
      }
      else if (height-1 == y) {
	p2 = location_R3(x, y-1);
	pDy = p1 - p2;
	pDy.normalize();
      }
      else {
	// check the pixel above
	p2 = location_R3(x, y-1);
	pDy = p1 - p2;
	pDy.normalize();
	// check the pixel below
	p2 = location_R3(x, y+1);
	diff = p2 - p1;
	diff.normalize();

	// Check whether discontinuity occurs.
	if (pDy * diff < 0.714) {  // > 45 degree
	  // pick the one with less difference in disparity
	  t  = 1.0 / getDisparity(x, y);
	  t1 = 1.0 / getDisparity(x, y-1);
	  t2 = 1.0 / getDisparity(x, y+1);
	  if (fabs(t-t1) < fabs(t-t2)) {
	    // pDy is already computed.
	  }
	  else {
	    pDy = diff;
	  }
	}
	else { 
	  // Combine both derivatives.
	  pDy += diff;
	  pDy.normalize();
	}
      }

      // Compute the normal from pDx and pDy
      // and make sure it's facing the viewer.
      N = pDx ^ pDy;
      N.normalize();
      viewDir = p1 - camera.COP;
      if (N * viewDir > 0) N = -N;
      pNormal = getNormal(x, y);  // get the pointer
      for (i=0; i<3; i++) {
	pNormal[i] = N[i];
      }
    } // for (x=...)
  } // for (y=...)
}

// Find the location in R^3 of the object sampled in pixel (x,y)
vec3 DepthImage::location_R3(int x, int y, float xOffset, float yOffset) const
{
  float disp = getDisparity(x, y);
  double t = 1.0/disp;
  vec3 dir, v;

  // Some softwares like the 3DS Max store the coordinates of the pixel
  // corner, thus requiring (xOffset, yOffset) such as (0.5, 0.5).
  dir = camera.a*(x+xOffset) + camera.b*(y+yOffset) + camera.c;

  // First, find the 3D location of pixel (x,y) which is
  // still on the projection plane. 
  // Then, map it to the real location by using disparity.
  v = camera.COP + dir*t;
  return v;
}


//===================================
//  class FrameBuffer
//===================================
FrameBuffer::FrameBuffer()
{
  width = height = 0;
  color = 0;
  Z = 0;
}

FrameBuffer::~FrameBuffer()
{
  if (color) delete[] color;
  if (Z)     delete[] Z;
}

void FrameBuffer::setSize(int w, int h, int noZ)
{
  int i;
  if (color) delete[] color;
  if (Z)     delete[] Z;

  width = w; height = h;
  color = new unsigned char[w*h*3];

  if (noZ)
    Z = 0;
  else
    Z = new float[w*h];

  for (i=0; i<w*h; i++) {
    color[3*i] = color[3*i+1] = color[3*i+2] = 0;
  }
  if (! noZ) for (i=0; i<w*h; i++) {
    Z[i] = FLT_MAX;
  }
}

void FrameBuffer::writeColor(int x, int y, unsigned char* RGB)
{
  if (! color) return;
  if (x < 0 || x >= width) return;
  if (y < 0 || y >= height) return;

  unsigned char* p = color + 3*(x+y*width);
  *(p++) = *(RGB++);
  *(p++) = *(RGB++);
  *(p++) = *(RGB++);
}

unsigned char* FrameBuffer::readColor(int x, int y)
{
  if (! color) return 0;
  if (x < 0 || x >= width) return 0;
  if (y < 0 || y >= height) return 0;
  return color + 3*(x+y*width);
}

void FrameBuffer::writeZ(int x, int y, float z)
{
  if (! Z) return;
  if (x < 0 || x >= width) return;
  if (y < 0 || y >= height) return;
  Z[x+y*width] = z;
}

float FrameBuffer::readZ(int x, int y)
{
  if (! Z) return 0;
  if (x < 0 || x >= width) return 0;
  if (y < 0 || y >= height) return 0;
  return Z[x+y*width];
}

int FrameBuffer::inside(int x, int y)
{
  return x >= 0 && x < width && y >= 0 && y < height;
}

void FrameBuffer::flipY()
{
  unsigned char* fb = new unsigned char[width*height*3];
  int x, y, yInv;
  
  // Reverse the Y direction in fb[]
  for (y=0; y<height; y++) {
    yInv = height - y - 1;
    // memcpy(&fb[yInv*3], &color[y*3], 3*width);
    for (x=0; x<width; x++) {
      fb[3*(x+yInv*width)] = color[3*(x+y*width)];
      fb[3*(x+yInv*width)+1] = color[3*(x+y*width)+1];
      fb[3*(x+yInv*width)+2] = color[3*(x+y*width)+2];
    }
  }
  for (y=0; y<height; y++) {
    for (x=0; x<width; x++) {
      color[3*(x+y*width)]   = fb[3*(x+y*width)];
      color[3*(x+y*width)+1] = fb[3*(x+y*width)+1];
      color[3*(x+y*width)+2] = fb[3*(x+y*width)+2];
    }
  }
  delete[] fb;
}

// Simply copy each pixel color from the depth image.
void FrameBuffer::from(const DepthImage& im)
{
  int x, y;
  unsigned char* rgb;

  setSize(im.width, im.height, 1);  // noZ == 1
  for (y=0; y<height; y++) {
    for (x=0; x<width; x++) {
      rgb = im.getColor(x, y);
      writeColor(x, y, rgb);
    }
  }
}

void FrameBuffer::from(unsigned char* rgb, int w, int h)
{
  setSize(w, h, 1);
  memcpy(color, rgb, 3*w*h);
}

void FrameBuffer::display()
{
  /* Setup the matrix for glRasterPos() and glDrawPixels() */
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(-0.5,0.5,-0.5,0.5,-1,1);
  glMatrixMode(GL_MODELVIEW);
  glRasterPos2f(-0.5, -0.5);
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
  glDisable(GL_DEPTH_TEST);

  glDrawPixels(width, height, GL_RGB, GL_UNSIGNED_BYTE, color);

  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();

  glutSwapBuffers();
  glEnable(GL_DEPTH_TEST);
}

