
/*****************************************************************************
*  ffactors.c
*
*  Refer to rad.h for details
*
*  Copyright (C) 1990-1991 Apple Computer, Inc.
*  All rights reserved.
*
*  12/1990 S. Eric Chen	
******************************************************************************/

#include "rad.h"
#include <math.h>
#include <stdlib.h>
#include <stdio.h>

extern void UVToXYZ(const TPoint3f quad[4], float u, float v, TPoint3f* xyz);

#define SAMPLE_RATE 4
#define PI	3.1415926

#define SubVector(c,a,b) (c).x=(a).x-(b).x, (c).y=(a).y-(b).y, (c).z=(a).z-(b).z
#define DotVector(a,b) ((a).x*(b).x + (a).y*(b).y + (a).z*(b).z)
#define NormalizeVector(n,a) 	((n)=sqrt(DotVector(a,a)), \
				(n)?((a).x/=n, (a).y/=n, (a).z/=n):0)

/* If the patch is rectangular, then we can find u and v easily
   by two projections. */
void XYZtoUV(
	TRadParams* params,
	TPatch* pPatch,
	TPoint3f p,
	double* u, double* v)
{
	TVector3f uVector, vVector, theVector;
	unsigned long i0, i1, i3;
	float tmp;

	i0 = pPatch->verts[0];
	i1 = pPatch->verts[1];
	i3 = pPatch->verts[3];
	
	SubVector(uVector, params->points[i3], params->points[i0]);
	SubVector(vVector, params->points[i1], params->points[i0]);
	SubVector(theVector, p, params->points[i0]);

	*u = DotVector(theVector, uVector) / DotVector(uVector, uVector);
	*v = DotVector(theVector, vVector) / DotVector(vVector, vVector);
}

/* Check if a patch is in the way of p1 to p2. To be used by
   visibility checking function -- Visible(). */
int Intersected(
		TRadParams* params,
		TPatch* pPatch,
		TPoint3f* p1, 
		TPoint3f* p2)
{
	TPoint3f* pCenter = &pPatch->center;
	TPoint3f* pNormal = &pPatch->normal;
	TPoint3f hit;
	double t, u, v;

	/* Represent the line as (1-t)*p1 + t*p2 = 0, find t
	   where the line intersects the plane of the patch. */
	t = ((p1->x - pCenter->x) * pNormal->x +
	     (p1->y - pCenter->y) * pNormal->y +
	     (p1->z - pCenter->z) * pNormal->z
	    ) /
	    ((p1->x - p2->x) * pNormal->x + 
	     (p1->y - p2->y) * pNormal->y + 
	     (p1->z - p2->z) * pNormal->z
	    );

	if (t <= 0.0 || t >= 1.0) return 0; 

	hit.x = (1-t) * p1->x + t * p2->x;	
	hit.y = (1-t) * p1->y + t * p2->y;	
	hit.z = (1-t) * p1->z + t * p2->z;

	XYZtoUV(params, pPatch, hit, &u, &v);
	return (u>=0.0 && u<=1.0 && v>=0.0 && v<=1.0);
}

/* Check visibility of p1 and p2. */
int Visible(
	TRadParams* params,
	TPoint3f* p1,
	TPoint3f* p2,
	TPatch* parent1,
	TPatch* parent2)
{
	TPatch* pPatch = params->patches;
	int i;

	for (i=params->nPatches; i--; pPatch++) {
		if (pPatch == parent1 || pPatch == parent2)
			continue;
		if (Intersected(params, pPatch, p1, p2))
			return 0;
	}

	return 1;
}
	
/* Compute form-factors from the shooting patch to every element */
void ComputeFormfactors(
		TRadParams *  params,
		double *	  formfactors,
		unsigned long shootPatch)
{
	unsigned long i, j, k;
	TPoint3f	center, pts[4];
	TVector3f	normal;
	TPatch*		sp;
	double*		fp;
	TElement*	ep;
	TVector3f	r;
	double		denom;
	double		rSquared;
	double		cosThetaE, cosThetaS;
	double		formFactorEstimate;
	double		dArea, u, v, du, dv;
	TPatch*		parent;

	/* get the center of shootPatch */
	sp = &(params->patches[shootPatch]);
	normal = sp->normal;
	
	/* misc setup */
	du = dv = 1.0/SAMPLE_RATE;
	dArea = sp->area/(SAMPLE_RATE*SAMPLE_RATE);
	for (i=0; i<4; i++)
		pts[i] = params->points[sp->verts[i]];	
	
	/* clear the formfactors */
	fp = formfactors;
	for (i=params->nElements; i--; fp++)
		*fp = 0.0;

	/*=================================================*/
	/* form factor calculation:                        */
	/* Divide the shooting patch into small elements,  */
	/* which are approximated by disks.                */
	/*=================================================*/
	
	ep = params->elements;
	fp = formfactors;
	for (i=params->nElements; i--; fp++, ep++)
	{

		if ( ep->patch == sp ) continue; /* get rid of triviality */

		/*------ A cheap but incorrect form factor ------*/
		/*------ Remove them when you're done.     ------*/
			SubVector( r , ep->center , sp->center );
			rSquared = DotVector( r , r );
			NormalizeVector( denom, r );
			cosThetaS = DotVector( r , sp->normal );
			cosThetaE = -DotVector( r , ep->normal );
			if (cosThetaE>0 && cosThetaS>0)
			*fp = ep->patch->area*cosThetaS * cosThetaE / (PI*rSquared);
			continue; 
		/*------ end of the incorrect version ------*/

		/*===========================================
		 * Your code start here.
		 * The following code is for your reference.
		 * Feel free to change or reuse any of them.
		 *===========================================*/

		parent = ep->patch;

		/* For each element, compute the differential to finite
		   form factor by sampling the shooting patch uniformly.
		   See "A Ray Tracing Algorithm for Progressive Radiosity"
		   by Wallace, Elmquist, & Haines.
		 */
		formFactorEstimate = 0.0;
		for (j=0, u=du/2.0; j<SAMPLE_RATE; j++, u+=du)
		    for (k=0, v=dv/2.0; k<SAMPLE_RATE; k++, v+=dv) {
			UVToXYZ(pts, u, v, &center);
			SubVector( r , ep->center , center );
			rSquared = DotVector( r , r );
			NormalizeVector( denom, r );
			cosThetaS = DotVector( r , normal );
			cosThetaE = -DotVector( r , ep->normal );
			
			/* back facing patch */
			if ((cosThetaS <= 0.0)||(cosThetaE <= 0.0)) {
				j = k = SAMPLE_RATE; /* break out two loops */
				break;
			}

			/* Skip if not visible from here. */
			if (! Visible(params, &center, &ep->center, sp, parent)) {
				continue;
			}

			/* Your code may look like this...
			formFactorEstimate += ......
			*/
				
		}
		*fp = dArea * formFactorEstimate;
	}
	/* It might be a good idea to print out and verify 
	   the sum of form factors of all elements here.   */
}
